<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Instagram Feed for Magento 2
 */

namespace Amasty\InstagramFeed\Model\Instagram;

use Amasty\InstagramFeed\Api\Data\PostInterface;
use Amasty\InstagramFeed\Model\ConfigProvider;
use Amasty\InstagramFeed\Model\Repository\PostRepository;
use Amasty\InstagramFeed\Model\Instagram\Video\DeleteOldVideoContent;
use Amasty\InstagramFeed\Model\ResourceModel\Post as PostResource;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Flag;
use Magento\Framework\FlagFactory;
use Magento\Framework\Stdlib\DateTime\DateTime;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;
use Psr\Log\LoggerInterface;

class Management
{
    public const POSTS_TO_SAVE = 180;

    public const FLAG_CODE = 'am_last_instagram_post_update_%d';

    /**
     * @var Client
     */
    private $client;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var PostResource
     */
    private $postResource;

    /**
     * @var PostRepository
     */
    private $postRepository;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var Flag|null
     */
    private $flag;

    /**
     * @var DateTime
     */
    private $dateTime;

    /**
     * @var FlagFactory
     */
    private $flagFactory;

    /**
     * @var IsMainInstagramSettingsDifferent
     */
    private $isMainInstagramSettingsDifferent;

    /**
     * @var DeleteOldVideoContent
     */
    private $deleteOldVideoContent;

    public function __construct(
        Client $client,
        PostResource $postResource,
        PostRepository $postRepository,
        StoreManagerInterface $storeManager,
        FlagFactory $flagFactory,
        DateTime $dateTime,
        LoggerInterface $logger,
        IsMainInstagramSettingsDifferent $isMainInstagramSettingsDifferent,
        DeleteOldVideoContent $deleteOldVideoContent,
        ConfigProvider $configProvider
    ) {
        $this->client = $client;
        $this->logger = $logger;
        $this->postResource = $postResource;
        $this->postRepository = $postRepository;
        $this->configProvider = $configProvider;
        $this->storeManager = $storeManager;
        $this->dateTime = $dateTime;
        $this->flagFactory = $flagFactory;
        $this->isMainInstagramSettingsDifferent = $isMainInstagramSettingsDifferent;
        $this->deleteOldVideoContent = $deleteOldVideoContent;
    }

    public function updateByCron()
    {
        $this->update();
    }

    public function update(?int $storeId = null, bool $skipLimitsCheck = false): void
    {
        try {
            if ($this->isUpdateAvailable($storeId, $skipLimitsCheck)) {
                $postData = $storeId !== null ? $this->getPosts($storeId) : $this->getAllPosts();
                if ($postData) {
                    $disabledPostIds = $this->postRepository->getAllDisabledPostIds();
                    $postData = $this->changeActivePostsStatus($postData, $disabledPostIds);
                    $this->postResource->replaceData($postData, $storeId);
                }
                $this->updateFlagData($storeId);
            }
        } catch (\Exception $exception) {
            $this->logger->error($exception->getMessage());
        }
    }

    private function changeActivePostsStatus(array $postData, array $disabledPostIds): array
    {
        if ($disabledPostIds) {
            $disabledPostsCount = count($disabledPostIds);
            $counter = 0;

            foreach ($postData as &$post) {
                if (in_array($post[PostInterface::IG_ID], $disabledPostIds)) {
                    $post[PostInterface::STATUS] = PostRepository::IS_DISABLED;
                    $counter++;
                }

                if ($counter >= $disabledPostsCount) {
                    break;
                }
            }
        }

        return $postData;
    }

    /**
     * @param int $storeId
     * @param array $oldPosts
     * @return array
     * @throws LocalizedException
     */
    public function getPosts($storeId, $oldPosts = [])
    {
        return array_merge(
            $this->client->loadPosts(self::POSTS_TO_SAVE, $storeId),
            $oldPosts
        );
    }

    /**
     * @throws LocalizedException
     */
    public function getAllPosts(): array
    {
        $postData = $this->getPosts(Store::DEFAULT_STORE_ID);

        foreach ($this->storeManager->getStores() as $store) {
            $storeId = (int)$store->getId();
            if ($this->isMainInstagramSettingsDifferent->execute($storeId)) {
                $postData = $this->getPosts($storeId, $postData);
            }
        }

        $this->deleteOldVideoContent();

        return $postData;
    }

    public function isUpdateAvailable(?int $storeId = null, bool $skipCheck = false): bool
    {
        return $skipCheck || ($this->getCurrentTime() - (int)$this->getFlag($storeId)->getFlagData()) / 3600 > 1;
    }

    private function updateFlagData(?int $storeId)
    {
        $this->getFlag($storeId)->setFlagData($this->getCurrentTime())->save();
    }

    public function resetFlagData(?int $storeId)
    {
        $this->getFlag($storeId)->setFlagData(null)->save();
    }

    /**
     * @return int
     */
    private function getCurrentTime(): int
    {
        return $this->dateTime->timestamp();
    }

    private function getFlag(?int $storeId): Flag
    {
        if ($this->flag === null) {
            try {
                $flag = $this->flagFactory->create(['data' => [
                    'flag_code' => sprintf(self::FLAG_CODE, $storeId)
                ]]);
                $this->flag = $flag->loadSelf();
            } catch (LocalizedException $e) {
                $this->logger->error($e->getMessage());
            }
        }

        return $this->flag;
    }

    private function deleteOldVideoContent(): void
    {
        foreach ($this->storeManager->getStores() as $store) {
            if ($this->configProvider->isVideoContentEnabled((int)$store->getId())) {
                $this->deleteOldVideoContent->execute();
                break;
            }
        }
    }
}
