<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Instagram Feed for Magento 2
 */

namespace Amasty\InstagramFeed\Model\Instagram;

use Amasty\InstagramFeed\Model\ConfigProvider;
use Amasty\InstagramFeed\Model\Instagram\Hashtag\GetHashtagInstagramIdsByName;
use Amasty\InstagramFeed\Model\Instagram\Hashtag\GetPostsByHashtag;
use Magento\Framework\Exception\LocalizedException;

class LoadMainMediaInformation
{
    public const MAX_POST_COUNT = 20;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var HttpCurl
     */
    private $httpCurl;

    /**
     * @var GetHashtagInstagramIdsByName
     */
    private $getHashtagInstagramIdsByName;

    /**
     * @var GetPostsByHashtag
     */
    private $postsByHashtag;

    /**
     * @var array
     */
    private $instagramFields = [];

    public function __construct(
        ConfigProvider $configProvider,
        HttpCurl $httpCurl,
        GetHashtagInstagramIdsByName $getHashtagInstagramIdsByName,
        GetPostsByHashtag $postsByHashtag,
        array $instagramFields
    ) {
        $this->configProvider = $configProvider;
        $this->httpCurl = $httpCurl;
        $this->getHashtagInstagramIdsByName = $getHashtagInstagramIdsByName;
        $this->postsByHashtag = $postsByHashtag;
        $this->instagramFields = implode(',', array_merge($this->instagramFields, $instagramFields));
    }

    /**
     * @throws LocalizedException
     */
    public function execute(int $count = self::MAX_POST_COUNT, ?int $storeId = null): array
    {
        if (!$this->configProvider->getAccessToken($storeId)) {
            return [];
        }

        if ($this->configProvider->isHashtagsEnabled($storeId)) {
            $hashtagIds = $this->getHashtagInstagramIdsByName->execute($storeId);

            return $this->postsByHashtag->execute($hashtagIds, $storeId);
        }

        $url = Client::ENDPOINT_URL . $this->configProvider->getUserId($storeId) . '/media?' .
            $this->httpCurl->getQueryString(
                [
                    ApiConstantsInterface::ACCESS_TOKEN => $this->configProvider->getAccessToken($storeId),
                    ApiConstantsInterface::LIMIT => $count,
                    ApiConstantsInterface::FIELDS => $this->instagramFields
                ]
            );
        $allMainMediaInfo = $this->httpCurl->loadByUrl($url);

        return array_slice($allMainMediaInfo, 0, $count);
    }
}
