<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Instagram Feed for Magento 2
 */

namespace Amasty\InstagramFeed\Model\Instagram;

use Amasty\Base\Model\Serializer;
use Laminas\Http\Request;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\HTTP\Adapter\Curl;
use Magento\Framework\Url\QueryParamsResolverInterface;
use Psr\Log\LoggerInterface;

class HttpCurl
{
    /**
     * @var Curl
     */
    private $curl;

    /**
     * @var Serializer
     */
    private $serializer;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var QueryParamsResolverInterface
     */
    private $queryParamsResolver;

    /**
     * @var array
     */
    private $headers = [];

    public function __construct(
        Curl $curl,
        Serializer $serializer,
        LoggerInterface $logger,
        QueryParamsResolverInterface $queryParamsResolver
    ) {
        $this->curl = $curl;
        $this->serializer = $serializer;
        $this->logger = $logger;
        $this->queryParamsResolver = $queryParamsResolver;
    }

    private function curl(string $url, string $method, array $bodyParams = []): string
    {
        // TODO: Replace in new magento versions (\Magento\Framework\HTTP\Adapter\Curl::setOptions)
        $this->curl->setConfig(['header' => false]);
        $this->curl->setOptions([CURLOPT_FOLLOWLOCATION => true]);
        $this->curl->write(
            $method,
            $url,
            '1.1',
            $this->getHeaders(),
            implode('&', $bodyParams)
        );
        $result = $this->curl->read();
        $this->curl->close();

        return $result;
    }

    /**
     * @param string $string
     * @return array|bool|float|int|mixed|string|null
     */
    private function unserialize(string $string)
    {
        try {
            return $this->serializer->unserialize($string);
        } catch (\Exception $exception) {
            $this->logger->error(__('Error occurred while getting the content from Instagram API: %1', $string));
            return [];
        }
    }

    public function getQueryString(array $params): string
    {
        $this->queryParamsResolver->addQueryParams($params);
        $query = $this->queryParamsResolver->getQuery();
        $this->queryParamsResolver->unsetData('query');
        $this->queryParamsResolver->unsetData('query_params');

        return $query;
    }

    /**
     * @throws LocalizedException
     */
    public function loadByUrl(string $url, string $method = Request::METHOD_GET): array
    {
        $result = $this->curl($url, $method);
        $result = $this->unserialize($result);
        $result = $result ?: [];

        if (isset($result['error']['message'])) {
            throw new LocalizedException(__($result['error']['message']));
        }

        return $result;
    }

    public function setHeaders(array $headers): self
    {
        $this->headers = $headers;

        return $this;
    }

    private function getHeaders(): array
    {
        $headers = [];
        foreach ($this->headers as $name => $value) {
            $headers[] = implode(': ', [$name, $value]);
        }

        return $headers;
    }
}
