<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Instagram Feed for Magento 2
 */

namespace Amasty\InstagramFeed\Model\Instagram\Hashtag;

use Amasty\InstagramFeed\Api\Data\PostInterface;
use Amasty\InstagramFeed\Model\ConfigProvider;
use Amasty\InstagramFeed\Model\Instagram\ApiConstantsInterface;
use Amasty\InstagramFeed\Model\Instagram\Client;
use Amasty\InstagramFeed\Model\Instagram\HttpCurl;
use Amasty\InstagramFeed\Model\Instagram\Management;
use Magento\Framework\Exception\LocalizedException;

class GetPostsByHashtag
{
    public const LIMIT_POSTS_BY_ONE_HASHTAG = 50;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var HttpCurl
     */
    private $httpCurl;

    /**
     * @var array
     */
    private $instagramFields = [];

    /**
     * @var array
     */
    private $videoExtensions = [];

    public function __construct(
        ConfigProvider $configProvider,
        HttpCurl $httpCurl,
        array $instagramFields,
        array $videoExtensions
    ) {
        $this->configProvider = $configProvider;
        $this->httpCurl = $httpCurl;
        $this->instagramFields = implode(',', array_merge($this->instagramFields, $instagramFields));
        $this->videoExtensions =  array_merge($this->videoExtensions, $videoExtensions);
    }

    /**
     * @throws LocalizedException
     */
    public function execute(array $hashtags, ?int $storeId): array
    {
        $sortingType = $this->configProvider->getSortOfHashtagPosts($storeId);
        $allMainMediaInfo = [];
        foreach ($hashtags as $hashtag) {
            $url = Client::ENDPOINT_URL . $hashtag . '/' . $sortingType . '_media?' .
                $this->httpCurl->getQueryString(
                    [
                        ApiConstantsInterface::ACCESS_TOKEN => $this->configProvider->getAccessToken($storeId),
                        ApiConstantsInterface::USER_ID => $this->configProvider->getUserId($storeId),
                        ApiConstantsInterface::FIELDS => $this->instagramFields,
                        ApiConstantsInterface::LIMIT => $this->getLimitPostsPerHashtag(count($hashtags))
                    ]
                );
            // phpcs:ignore Magento2.Performance.ForeachArrayMerge.ForeachArrayMerge
            $allMainMediaInfo = array_merge($allMainMediaInfo, $this->httpCurl->loadByUrl($url)['data']);
        }

        $this->setChildrenImageAsMainMediaUrl($allMainMediaInfo);

        return [
            'data' => $allMainMediaInfo
        ];
    }

    private function getLimitPostsPerHashtag(int $hashtagsCount): int
    {
        $limitPerOneHashtag = Management::POSTS_TO_SAVE / $hashtagsCount;

        return min($limitPerOneHashtag, self::LIMIT_POSTS_BY_ONE_HASHTAG);
    }

    private function setChildrenImageAsMainMediaUrl(array &$mediaData): void
    {
        foreach ($mediaData as &$media) {
            if (isset($media['children']['data'])) {
                foreach ($media['children']['data'] as $childrenItem) {
                    // phpcs:disable Magento2.Functions.DiscouragedFunction.Discouraged
                    $path = parse_url($childrenItem[PostInterface::MEDIA_URL], PHP_URL_PATH);
                    // phpcs:disable Magento2.Functions.DiscouragedFunction.DiscouragedWithAlternative
                    $extension = pathinfo($path, PATHINFO_EXTENSION);
                    if (!in_array($extension, $this->videoExtensions)) {
                        $media[PostInterface::MEDIA_URL] = $childrenItem[PostInterface::MEDIA_URL];
                        break;
                    }
                }
                unset($media['children']);
            }
        }
    }
}
