<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Instagram Feed for Magento 2
 */

namespace Amasty\InstagramFeed\Model\Instagram\Hashtag;

use Amasty\InstagramFeed\Api\Data\HashtagInterface;
use Amasty\InstagramFeed\Model\ConfigProvider;
use Amasty\InstagramFeed\Model\Instagram\ApiConstantsInterface;
use Amasty\InstagramFeed\Model\Instagram\Client;
use Amasty\InstagramFeed\Model\Instagram\HttpCurl;
use Amasty\InstagramFeed\Model\ResourceModel\Hashtag;
use Magento\Framework\Exception\LocalizedException;

class GetHashtagInstagramIdsByName
{
    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var HttpCurl
     */
    private $httpCurl;

    /**
     * @var Hashtag
     */
    private $hashtagResource;

    /**
     * @var GetHashtagInfo
     */
    private $hashtagInfo;

    public function __construct(
        ConfigProvider $configProvider,
        HttpCurl $httpCurl,
        Hashtag $hashtagResource,
        GetHashtagInfo $hashtagInfo
    ) {
        $this->configProvider = $configProvider;
        $this->httpCurl = $httpCurl;
        $this->hashtagResource = $hashtagResource;
        $this->hashtagInfo = $hashtagInfo;
    }

    /**
     * @throws LocalizedException
     * @throws \Exception
     */
    public function execute(?int $storeId, ?array $hashtags = null): array
    {
        $hashtags = $hashtags ?: $this->configProvider->getHashtags($storeId);

        list(
            GetHashtagInfo::NEED_GET_FROM_API_HASHTAGS => $toGetFromApiHashtags,
            GetHashtagInfo::DATABASE_HASHTAGS => $hashtagsWithInstagramIds
            ) = $this->hashtagInfo->execute($hashtags);

        $hashtagApiInfo = [];
        if (!empty($toGetFromApiHashtags)) {
            $hashtagApiInfo = $this->getHashtagsInfo($toGetFromApiHashtags, $storeId);
            if (!empty($hashtagApiInfo)) {
                $this->hashtagResource->addNewData($this->prepareHashtagData($hashtagApiInfo));
            }
        }

        return array_merge($hashtagsWithInstagramIds, $hashtagApiInfo);
    }

    private function getHashtagsInfo(array $hashtags, ?int $storeId): array
    {
        $hashtagInfo = [];
        foreach ($hashtags as $hashtag) {
            $url = Client::ENDPOINT_URL . 'ig_hashtag_search?' .
                $this->httpCurl->getQueryString(
                    [
                        ApiConstantsInterface::ACCESS_TOKEN => $this->configProvider->getAccessToken($storeId),
                        ApiConstantsInterface::USER_ID => $this->configProvider->getUserId($storeId),
                        ApiConstantsInterface::QUERY => $hashtag
                    ]
                );

            try {
                $data = $this->httpCurl->loadByUrl($url);
            } catch (LocalizedException $exception) {
                continue;
            }

            $hashtagInfo[$hashtag] = $data['data'][0]['id'];
        }

        return $hashtagInfo;
    }

    private function prepareHashtagData(array $hashtags): array
    {
        $hashtagData = [];
        foreach ($hashtags as $key => $hashtag) {
            $hashtagData[] = [
                HashtagInterface::NAME => $key,
                HashtagInterface::HASHTAG_ID => $hashtag
            ];
        }

        return $hashtagData;
    }
}
