<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Instagram Feed for Magento 2
 */

namespace Amasty\InstagramFeed\Model\Instagram;

use Amasty\Base\Model\Serializer;
use Amasty\InstagramFeed\Api\Data\PostInterface;
use Amasty\InstagramFeed\Model\Backend\GetInternalToken;
use Amasty\InstagramFeed\Model\ConfigProvider;
use Amasty\InstagramFeed\Model\Instagram\Operation\MediaPathResolver;
use Amasty\InstagramFeed\Model\Repository\PostRepository;
use Magento\Framework\Exception\ConfigurationMismatchException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Stdlib\DateTime;
use Magento\Framework\Stdlib\DateTime\TimezoneInterface;
use Magento\Framework\UrlInterface;
use Magento\Store\Model\StoreManagerInterface;
use Psr\Log\LoggerInterface;

/**
 * Class Client
 * Class implements communication between Magento and Instagram
 */
class Client
{
    public const AUTHORIZE_URL_PATH = 'socialconnect/instagram/authorize';
    public const ENDPOINT_URL = 'https://graph.facebook.com/v8.0/';
    public const ACCOUNTS_URL = self::ENDPOINT_URL . 'me/accounts/';
    public const EMBED_URL = 'https://graph.facebook.com/v9.0/instagram_oembed?';

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var Serializer
     */
    private $serializer;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var GetInternalToken
     */
    private $getInternalToken;

    /**
     * @var TimezoneInterface
     */
    private $timezone;

    /**
     * @var HttpCurl
     */
    private $httpCurl;

    /**
     * @var MediaPathResolver
     */
    private $mediaPathResolver;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var LoadMainMediaInformation
     */
    private $loadMainMediaInformation;

    public function __construct(
        Serializer $serializer,
        ConfigProvider $configProvider,
        StoreManagerInterface $storeManager,
        GetInternalToken $getInternalToken,
        TimezoneInterface $timezone,
        HttpCurl $httpCurl,
        MediaPathResolver $mediaPathResolver,
        LoggerInterface $logger,
        LoadMainMediaInformation $loadMainMediaInformation
    ) {
        $this->configProvider = $configProvider;
        $this->serializer = $serializer;
        $this->storeManager = $storeManager;
        $this->getInternalToken = $getInternalToken;
        $this->timezone = $timezone;
        $this->httpCurl = $httpCurl;
        $this->mediaPathResolver = $mediaPathResolver;
        $this->logger = $logger;
        $this->loadMainMediaInformation = $loadMainMediaInformation;
    }

    /**
     * @throws LocalizedException
     */
    public function loadPosts(int $count, ?int $storeId = null): array
    {
        $result = [];
        $mediaCount = 0;
        $allMainMediaInfo = $this->loadMainMediaInformation->execute($count, $storeId);

        if (empty($allMainMediaInfo['data'])) {
            return $result;
        }

        foreach ($allMainMediaInfo['data'] as $mediaData) {
            if (empty($mediaData['id'])) {
                continue;
            }

            $mediaData[PostInterface::IG_ID] = $mediaData['id'];
            unset($mediaData['id']);

            try {
                $mediaData[PostInterface::MEDIA_URL] =
                    $this->mediaPathResolver->getMediaUrl($mediaData, $storeId);
                if (!$mediaData[PostInterface::MEDIA_URL]) {
                    continue;
                }
            } catch (ConfigurationMismatchException $exception) {
                continue;
            } catch (LocalizedException $exception) {
                $this->logger->error(__('Unable to process the file URL: %1', $exception->getMessage()));
                continue;
            }

            unset($mediaData['thumbnail_url']);

            // prepare for insert into index table
            if ($storeId !== null) {
                $mediaData[PostInterface::STORE_ID] = $storeId;
            }

            $mediaData[PostInterface::TIMESTAMP] =
                $this->prepareTimestampFormat($mediaData[PostInterface::TIMESTAMP]);
            $mediaData[PostInterface::CAPTION] = $mediaData[PostInterface::CAPTION] ?? '';
            $mediaData[PostInterface::STATUS] = PostRepository::IS_ENABLED;
            $mediaData[PostInterface::SHORTCODE] = $mediaData[PostInterface::SHORTCODE] ?? '';
            $mediaData[PostInterface::LIKE_COUNT] = $mediaData[PostInterface::LIKE_COUNT] ?? null;
            $result[] = $mediaData;
            $mediaCount++;

            if ($mediaCount >= $count) {
                break;
            }
        }

        return $result;
    }

    /**
     * @param $postUrl
     * @param $maxWidth
     * @param $hideCaption
     * @return string
     * @throws LocalizedException
     */
    public function loadSinglePostHtml($postUrl, $maxWidth, $hideCaption)
    {
        $url = self::EMBED_URL
            . $this->httpCurl->getQueryString([
                ApiConstantsInterface::POST_URL => $postUrl,
                ApiConstantsInterface::ACCESS_TOKEN => $this->configProvider->getAccessToken(),
                ApiConstantsInterface::MAX_WIDTH => $maxWidth,
                ApiConstantsInterface::HIDE_CAPTION => $hideCaption
            ]);

        $result = $this->httpCurl->loadByUrl($url);

        return $result['html'] ?? '';
    }

    /**
     * @throws LocalizedException
     */
    public function getUserIdByToken(string $token): ?string
    {
        $userId = null;
        $postData = [
            ApiConstantsInterface::FIELDS => 'id,connected_instagram_account',
            ApiConstantsInterface::ACCESS_TOKEN => $token
        ];

        $url = self::ACCOUNTS_URL . '?' . http_build_query($postData);
        $result = $this->httpCurl->loadByUrl($url);

        if (isset($result['data']) && is_array($result['data'])) {
            foreach ($result['data'] as $account) {
                if (isset($account['connected_instagram_account']['id'])) {
                    $userId = $account['connected_instagram_account']['id'];
                    break;
                }
            }
        } elseif (isset($result['error']['message'])) {
            throw new LocalizedException(__($result['error']['message']));
        }

        return $userId;
    }

    /**
     * @param $storeId
     * @return string
     * @throws NoSuchEntityException
     */
    public function getAuthorizeUrl($storeId)
    {
        return $this->configProvider->getAuthorizeHost()
            . self::AUTHORIZE_URL_PATH
            . '?'
            . http_build_query(['key' => $this->generateAuthorizationKey($storeId)]);
    }

    /**
     * @return string
     * @throws NoSuchEntityException
     */
    private function getRefererUrl()
    {
        $storeId = $this->storeManager->getDefaultStoreView()->getId();
        return $this->storeManager->getStore($storeId)->getBaseUrl(UrlInterface::URL_TYPE_LINK, true);
    }

    /**
     * @param int $storeId
     * @return string
     * @throws NoSuchEntityException
     */
    private function generateAuthorizationKey($storeId)
    {
        $data = [
            'referer' => $this->getRefererUrl(),
            'store_id' => $storeId,
            'internal_token' => $this->getInternalToken->execute()
        ];
        return base64_encode($this->serializer->serialize($data));
    }

    /**
     * @return bool
     * @throws NoSuchEntityException
     */
    public function getIsRefererSecure()
    {
        return strpos($this->getRefererUrl(), 'https') === 0;
    }

    private function prepareTimestampFormat(string $timestamp): string
    {
        return $this->timezone->date($timestamp)->format(DateTime::DATETIME_PHP_FORMAT);
    }
}
