<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Instagram Feed for Magento 2
 */

namespace Amasty\InstagramFeed\Model;

use Amasty\Base\Model\ConfigProviderAbstract;
use Magento\Framework\App\Config\ReinitableConfigInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\Config\Storage\WriterInterface;
use Magento\Framework\Encryption\EncryptorInterface;
use Magento\Store\Model\ScopeInterface;

/**
 * Class ConfigProvider
 *
 * Class provides configs for the extension
 */
class ConfigProvider extends ConfigProviderAbstract
{
    /**
     * @var string
     */
    protected $pathPrefix = 'aminstagramfeed/';

    /**#@+
     * Constants defined for xpath of system configuration
     */
    public const XPATH_ENABLED = 'general/enabled';
    public const XPATH_DOWNLOAD_VIDEO = 'video/download_video';
    public const XPATH_DOWNLOAD_VIDEO_MAX_SIZE = 'video/max_download_video_size';
    public const XPATH_INSTAGRAM_SESSION_ID = 'video/instagram_session_id';
    public const XPATH_HASHTAGS_ENABLED = 'hashtags/load_posts_by_hashtags';
    public const XPATH_HASHTAGS = 'hashtags/hashtags';
    public const XPATH_HASHTAGS_SORT = 'hashtags/posts_sort';
    public const XPATH_INTERNAL_TOKEN = 'credentials/internal_token';
    public const CLIENT_ID = 'credentials/client_id';
    public const CLIENT_SECRET = 'credentials/client_secret';
    public const ACCESS_TOKEN = 'credentials/access_token_new';
    public const USER_ID = 'credentials/user_id';
    public const AUTHORIZE_HOST = 'credentials/authorize_host';

    /**
     * @var WriterInterface
     */
    private $configWriter;

    /**
     * @var ReinitableConfigInterface
     */
    private $reinitableConfig;

    /**
     * @var EncryptorInterface
     */
    private $encryptor;

    public function __construct(
        ReinitableConfigInterface $reinitableConfig,
        ScopeConfigInterface $scopeConfig,
        WriterInterface $configWriter,
        EncryptorInterface $encryptor
    ) {
        parent::__construct($scopeConfig);
        $this->configWriter = $configWriter;
        $this->reinitableConfig = $reinitableConfig;
        $this->encryptor = $encryptor;
    }

    /**
     * @inheritdoc
     */
    public function isEnabled()
    {
        return $this->isSetFlag(self::XPATH_ENABLED);
    }

    /**
     * @param null|int $storeId
     * @return mixed|string
     */
    public function getAccessToken($storeId = null)
    {
        if ($accessToken = $this->getValue(self::ACCESS_TOKEN, $storeId)) {
            $accessToken = $this->encryptor->decrypt($accessToken);
        }

        return $accessToken;
    }

    /**
     * @param int|null $storeId
     * @return mixed
     */
    public function getUserId($storeId = null)
    {
        return $this->getValue(self::USER_ID, $storeId);
    }

    /**
     * @param string $accessToken
     * @param null|int $storeId
     *
     * @return $this;
     */
    public function saveAccessToken($accessToken, $storeId = null)
    {
        $accessToken = $this->encryptor->encrypt($accessToken);
        if ($storeId) {
            $this->getConfigWriter()->save(
                $this->pathPrefix . self::ACCESS_TOKEN,
                $accessToken,
                ScopeInterface::SCOPE_STORES,
                $storeId
            );
        } else {
            $this->getConfigWriter()->save($this->pathPrefix . self::ACCESS_TOKEN, $accessToken);
        }

        $this->getReinitableConfig()->reinit();

        return $this;
    }

    /**
     * @param $userId
     * @param null $storeId
     *
     * @return $this
     */
    public function saveUserId($userId, $storeId = null)
    {
        if ($storeId) {
            $this->getConfigWriter()->save(
                $this->pathPrefix . self::USER_ID,
                $userId,
                ScopeInterface::SCOPE_STORES,
                $storeId
            );
        } else {
            $this->getConfigWriter()->save($this->pathPrefix . self::USER_ID, $userId);
        }

        $this->getReinitableConfig()->reinit();

        return $this;
    }

    public function removeAccessToken(?int $storeId = null): void
    {
        if ($storeId) {
            $this->getConfigWriter()->delete(
                $this->pathPrefix . self::ACCESS_TOKEN,
                ScopeInterface::SCOPE_STORES,
                $storeId
            );
        } else {
            $this->getConfigWriter()->delete($this->pathPrefix . self::ACCESS_TOKEN);
        }

        $this->getReinitableConfig()->reinit();
    }

    public function removeUserId(?int $storeId = null): void
    {
        if ($storeId) {
            $this->getConfigWriter()->delete(
                $this->pathPrefix . self::USER_ID,
                ScopeInterface::SCOPE_STORES,
                $storeId
            );
        } else {
            $this->getConfigWriter()->delete($this->pathPrefix . self::USER_ID);
        }

        $this->getReinitableConfig()->reinit();
    }

    /**
     * @return string
     */
    public function getPathPrefix()
    {
        return $this->pathPrefix;
    }

    /**
     * @return WriterInterface
     */
    public function getConfigWriter()
    {
        return $this->configWriter;
    }

    /**
     * @return ReinitableConfigInterface
     */
    public function getReinitableConfig()
    {
        return $this->reinitableConfig;
    }

    /**
     * @return mixed
     */
    public function getAuthorizeHost()
    {
        return $this->getValue(self::AUTHORIZE_HOST);
    }

    /**
     * @param bool $needGenerate = false
     * @return string
     */
    public function getInternalToken($needGenerate = false)
    {
        $token = $this->getValue(self::XPATH_INTERNAL_TOKEN);
        if ($needGenerate) {
            $token = hash('sha256', rand(1, PHP_INT_MAX));
            $this->getConfigWriter()->save($this->pathPrefix . self::XPATH_INTERNAL_TOKEN, $token);
            $this->getReinitableConfig()->reinit();
        }

        return $token;
    }

    public function isVideoContentEnabled(?int $storeId = null): bool
    {
        return $this->isSetFlag(self::XPATH_DOWNLOAD_VIDEO, $storeId);
    }

    public function getMaxDownloadVideoSize(?int $storeId): int
    {
        return (int)$this->getValue(self::XPATH_DOWNLOAD_VIDEO_MAX_SIZE, $storeId);
    }

    public function getSessionId(): string
    {
        if ($sessionId = $this->getValue(self::XPATH_INSTAGRAM_SESSION_ID)) {
            $sessionId = $this->encryptor->decrypt($sessionId);
        }

        return (string)$sessionId;
    }

    public function isHashtagsEnabled(?int $storeId = null): bool
    {
        return $this->isSetFlag(self::XPATH_HASHTAGS_ENABLED, $storeId);
    }

    public function getSortOfHashtagPosts(?int $storeId = null): string
    {
        return (string)$this->getValue(self::XPATH_HASHTAGS_SORT, $storeId);
    }

    public function getHashtags(?int $storeId = null): array
    {
        $hashtagString = (string)$this->getValue(self::XPATH_HASHTAGS, $storeId);

        return explode(',', $hashtagString);
    }
}
