<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Instagram Feed for Magento 2
 */

namespace Amasty\InstagramFeed\Model\Config\Backend;

use Amasty\InstagramFeed\Model\Instagram\Hashtag\GetHashtagInstagramIdsByName;
use Magento\Framework\App\Cache\TypeListInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\Config\Value;
use Magento\Framework\Data\Collection\AbstractDb;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Message\ManagerInterface as MessageManagerInterface;
use Magento\Framework\Model\Context;
use Magento\Framework\Model\ResourceModel\AbstractResource;
use Magento\Framework\Registry;

class HashtagsValidator extends Value
{
    /**
     * @var MessageManagerInterface
     */
    private $messageManager;

    /**
     * @var DateValidator
     */
    private $dateValidator;

    /**
     * @var GetHashtagInstagramIdsByName
     */
    private $getHashtagInstagramIdsByName;

    public function __construct(
        Context $context,
        Registry $registry,
        ScopeConfigInterface $config,
        TypeListInterface $cacheTypeList,
        MessageManagerInterface $messageManager,
        DateValidator $dateValidator,
        GetHashtagInstagramIdsByName $getHashtagInstagramIdsByName,
        AbstractResource $resource = null,
        AbstractDb $resourceCollection = null,
        array $data = []
    ) {
        $this->messageManager = $messageManager;
        $this->dateValidator = $dateValidator;
        $this->getHashtagInstagramIdsByName = $getHashtagInstagramIdsByName;

        parent::__construct(
            $context,
            $registry,
            $config,
            $cacheTypeList,
            $resource,
            $resourceCollection,
            $data
        );
    }

    /**
     * @throws LocalizedException
     */
    public function beforeSave(): Value
    {
        if (!$this->isValueChanged()) {
            return parent::beforeSave();
        }

        $hashtagsString = trim($this->getValue());
        $hashtags = explode(',', $hashtagsString);

        foreach ($hashtags as $key => &$hashtag) {
            $hashtag = trim($hashtag);
            $hashtag = ltrim($hashtag, '#');

            if (!strlen($hashtag)) {
                unset($hashtags[$key]);
            }

            $hashtag = strtolower($hashtag);
        }

        $this->checkIsEmpty($hashtags);

        $this->dateValidator->validate($hashtags);

        $validatedHashtags = array_flip($this->getHashtagInstagramIdsByName->execute(null, $hashtags));
        $notValidatedHashtags = array_diff($hashtags, $validatedHashtags);
        $hashtags = array_diff($hashtags, $notValidatedHashtags);

        $validatedHashtagString = implode(',', $hashtags);
        if ($validatedHashtagString !== $hashtagsString) {
            if (!empty($notValidatedHashtags)) {
                $this->messageManager->addWarningMessage(
                    __(
                        'Hashtag(s) %1 does (do) not exist in Instagram and is (are) deleted from the config',
                        implode(',', $notValidatedHashtags)
                    )
                );
            }
        }

        $this->checkIsEmpty($hashtags);

        $this->setValue($validatedHashtagString);

        return parent::beforeSave();
    }

    /**
     * @throws LocalizedException
     */
    private function checkIsEmpty(array $hashtags): void
    {
        if (empty($hashtags)) {
            throw new LocalizedException(__('Hashtag string can not be empty'));
        }
    }
}
