<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Instagram Feed for Magento 2
 */

namespace Amasty\InstagramFeed\Model\Config\Backend;

use Amasty\InstagramFeed\Model\Instagram\Hashtag\GetHashtagInfo;
use Amasty\InstagramFeed\Model\ResourceModel\Hashtag;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Stdlib\DateTime;
use Magento\Framework\Stdlib\DateTime\TimezoneInterface;

class DateValidator
{
    public const LIMIT_HASHTAGS_PER_WEEK = 30;

    /**
     * @var Hashtag
     */
    private $hashtagResource;

    /**
     * @var TimezoneInterface
     */
    private $timezone;

    /**
     * @var GetHashtagInfo
     */
    private $hashtagInfo;

    public function __construct(
        Hashtag $hashtagResource,
        TimezoneInterface $timezone,
        GetHashtagInfo $hashtagInfo
    ) {
        $this->hashtagResource = $hashtagResource;
        $this->timezone = $timezone;
        $this->hashtagInfo = $hashtagInfo;
    }

    /**
     * @throws LocalizedException
     */
    public function validate(array $hashtags): void
    {
        list(GetHashtagInfo::NEED_GET_FROM_API_HASHTAGS => $unsavedHashtags) = $this->hashtagInfo->execute($hashtags);
        $hashtagCountPerWeek = $this->hashtagResource->getCountHashtagsPerWeekLimit();

        if ($hashtagCountPerWeek >= self::LIMIT_HASHTAGS_PER_WEEK) {
            throw new LocalizedException(
                __(
                    'Currently, you can\'t add all new hashtags due to a limitation in the Instagram API. '
                    . 'You can add a hashtag from %1. You can also use previously generated hashtags '
                    . '(Amasty->InstagramFeed->Hashtags)',
                    $this->prepareDate($this->hashtagResource->getNextDateToGenerateHashtag())
                )
            );
        }

        if ($hashtagCountPerWeek + count($unsavedHashtags) > self::LIMIT_HASHTAGS_PER_WEEK) {
            throw new LocalizedException(
                __(
                    'You can add only %1 hashtag(s) due to a limitation in the Instagram API. '
                    . 'You added %2 unique hashtags within a 7-day period.'
                    . ' You can also use previously generated hashtags (Amasty->InstagramFeed->Hashtags)',
                    self::LIMIT_HASHTAGS_PER_WEEK - $hashtagCountPerWeek,
                    $hashtagCountPerWeek
                )
            );
        }
    }

    private function prepareDate(string $date): string
    {
        return $this->timezone->date($date)->format(DateTime::DATETIME_PHP_FORMAT);
    }
}
