<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Follow Up Email for Magento 2
 */

namespace Amasty\Followup\Ui\DataProvider\Rule\Modifier;

use Amasty\Followup\Model\Rule;
use Amasty\Followup\Model\Rule\Locator;
use Amasty\Followup\Model\Source\Rule\StartEventType;
use Magento\Store\Model\StoreManagerInterface as StoreManager;
use Magento\Ui\DataProvider\Modifier\ModifierInterface;
use Magento\Framework\Stdlib\ArrayManager;

class General implements ModifierInterface
{
    /**
     * @var Locator
     */
    private $ruleLocator;

    /**
     * @var StoreManager
     */
    private $storeManager;

    /**
     * @var StartEventType
     */
    private $startEventType;

    /**
     * @var ArrayManager
     */
    private $arrayManager;

    public function __construct(
        Locator $ruleLocator,
        StoreManager $storeManager,
        StartEventType $startEventType,
        ArrayManager $arrayManager
    ) {
        $this->ruleLocator = $ruleLocator;
        $this->storeManager = $storeManager;
        $this->startEventType = $startEventType;
        $this->arrayManager = $arrayManager;
    }

    public function modifyData(array $data): array
    {
        $data['isEditForm'] = (bool)$this->ruleLocator->getRule()->getRuleId();
        $data['isNotSingleStoreMode'] = !$this->storeManager->isSingleStoreMode();

        return $data;
    }

    public function modifyMeta(array $meta): array
    {
        $this
            ->disableComponentsForNewRule($meta)
            ->disableByEventType($meta);

        return $meta;
    }

    private function disableComponentsForNewRule(array &$meta): self
    {
        $rule = $this->ruleLocator->getRule();
        if (!$rule->getRuleId()) {
            $paths = [
                'general_fieldset/children/customer_date_event/arguments/data/config',
                'general_fieldset/children/cancel_event_type/arguments/data/config',
                'general_fieldset/children/to_subscribers/arguments/data/config',
                'general_fieldset/children/is_active/arguments/data/config',
                'stores_customer_groups_fieldset/arguments/data/config',
                'sender_details_fieldset/arguments/data/config',
                'google_analytics_fieldset/arguments/data/config',
                'segments_fieldset/arguments/data/config',
                'schedule_fieldset/arguments/data/config',
                'conditions_fieldset/arguments/data/config',
                'customer_fieldset/arguments/data/config',
                'order_fieldset/arguments/data/config'
            ];
            $this->disablePaths($paths, $meta);
        }

        return $this;
    }

    private function disableByEventType(array &$meta): self
    {
        $paths = [];
        $rule = $this->ruleLocator->getRule();
        if ($rule->getRuleId()) {
            if ($rule->getStartEventType() != Rule::TYPE_CUSTOMER_DATE) {
                $paths[] = 'general_fieldset/children/customer_date_event/arguments/data/config';
            }
            if ($rule->getStartEventType() != Rule::TYPE_CUSTOMER_SUBSCRIPTION) {
                $paths[] = 'general_fieldset/children/to_subscribers/arguments/data/config';
            }
            if ($this->startEventType->isOrderRelated($rule->getStartEventType())) {
                $paths[] = 'customer_fieldset/arguments/data/config';
            } else {
                $paths[] = 'order_fieldset/arguments/data/config';
                $paths[] = 'conditions_fieldset/arguments/data/config';
            }

            $this->disablePaths($paths, $meta);
        }

        return $this;
    }

    private function disablePaths(array $paths, array &$meta): self
    {
        $config = ['componentDisabled' => true];
        foreach ($paths as $path) {
            $meta = $this->arrayManager->set($path, $meta, $config);
        }

        return $this;
    }
}
