<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Follow Up Email for Magento 2
 */

namespace Amasty\Followup\Plugin;

use Amasty\Followup\Model\Rule;
use Amasty\Followup\Model\ScheduleFactory;
use Magento\Catalog\Model\Product;
use Magento\Customer\Model\ResourceModel\CustomerRepository;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Module\Manager;
use Magento\Framework\ObjectManagerInterface;
use Magento\Wishlist\Model\ResourceModel\Wishlist\CollectionFactory as WishlistCollectionFactory;

class SaveProduct
{
    /**
     * @var ScheduleFactory
     */
    private $scheduleFactory;

    /**
     * @var CustomerRepository
     */
    private $customerRepository;

    /**
     * @var WishlistCollectionFactory
     */
    private $wishListCollectionFactory;

    /**
     * @var bool
     */
    private $isMagentoWishlistEnabled = false;

    /**
     * @var Manager
     */
    private $moduleManager;

    /**
     * @var ObjectManagerInterface
     */
    private $objectManager;

    public function __construct(
        ScheduleFactory $scheduleFactory,
        CustomerRepository $customerRepository,
        Manager $moduleManager,
        ObjectManagerInterface $objectManager
    ) {
        $this->scheduleFactory = $scheduleFactory;
        $this->customerRepository = $customerRepository;
        $this->moduleManager = $moduleManager;
        $this->objectManager = $objectManager;

        $this->initMagentoWishlistCollectionFactory();
    }

    public function afterBeforeSave(Product $subject): Product
    {
        if (!$this->isMagentoWishlistEnabled || !$subject->getId()) {
            return $subject;
        }

        $oldData = $subject->getOrigData();

        $oldSpecialPrice = $oldData['special_price'] ?? null;
        $newSpecialPrice = $subject->getSpecialPrice();

        $oldSpecialFromDate = $oldData['special_from_date'] ?? null;
        $newSpecialFromDate = $subject->getSpecialFromDate();

        $oldSpecialToDate = $oldData['special_to_date'] ?? null;
        $newSpecialToDate = $subject->getSpecialToDate();

        $onSale = false;
        $backInStock = false;

        if ($oldSpecialPrice != $newSpecialPrice
            || $oldSpecialFromDate != $newSpecialFromDate
            || $oldSpecialToDate != $newSpecialToDate
        ) {
            $onSale = true;
        }

        $oldIsInStock = $oldData['quantity_and_stock_status']['is_in_stock'] ?? null;
        $qtyAndStock = $subject->getQuantityAndStockStatus();
        $newIsInStock = $qtyAndStock['is_in_stock'] ?? null;

        if ($oldIsInStock != $newIsInStock && $newIsInStock == 1) {
            $backInStock = true;
        }

        if ($onSale || $backInStock) {
            $types = [];

            if ($onSale) {
                $types[] = Rule::TYPE_CUSTOMER_WISHLIST_SALE;
            } elseif ($backInStock) {
                $types[] = Rule::TYPE_CUSTOMER_WISHLIST_BACK_INSTOCK;
            }

            $wishListCollection = $this->wishListCollectionFactory->create();
            $wishListCollection->getSelect()->joinLeft(
                ['wishlist_item' => $wishListCollection->getTable('wishlist_item')],
                'main_table.wishlist_id = wishlist_item.wishlist_id',
                []
            );
            $wishListCollection->getSelect()->where('wishlist_item.product_id = ' . $subject->getId());

            foreach ($wishListCollection as $wishItem) {
                try {
                    $customer = $this->customerRepository->getById((int)$wishItem->getCustomerId());
                    $this->scheduleFactory->create()->checkCustomerRules($customer, $types, $subject);
                } catch (NoSuchEntityException $e) {
                    null;
                }
            }
        }

        return $subject;
    }

    private function initMagentoWishlistCollectionFactory(): void
    {
        if ($this->moduleManager->isEnabled('Magento_Wishlist')) {
            $this->wishListCollectionFactory = $this->objectManager->get(WishlistCollectionFactory::class);
            $this->isMagentoWishlistEnabled = true;
        }
    }
}
