<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Follow Up Email for Magento 2
 */

namespace Amasty\Followup\Observer;

use Amasty\Followup\Model\Rule as Rule;
use Amasty\Followup\Model\ScheduleFactory;
use Magento\Customer\Model\CustomerFactory;
use Magento\Customer\Model\GroupManagement;
use Magento\Framework\Event\Observer as EventObserver;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Stdlib\DateTime\DateTime;
use Magento\Newsletter\Model\Subscriber;
use Magento\Store\Model\StoreManagerInterface;

class NewsletterSubscriber implements ObserverInterface
{
    /**
     * @var bool
     */
    private static $onNewsletterSubscriberSaveAfterChecked = false;

    /**
     * @var bool
     */
    public static $isFirstSubscribe = false;

    /**
     * @var ScheduleFactory
     */
    private $scheduleFactory;

    /**
     * @var CustomerFactory
     */
    private $customerFactory;

    /**
     * @var DateTime
     */
    private $date;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    public function __construct(
        ScheduleFactory $scheduleFactory,
        CustomerFactory $customerFactory,
        DateTime $date,
        StoreManagerInterface $storeManager
    ) {
        $this->scheduleFactory = $scheduleFactory;
        $this->customerFactory = $customerFactory;
        $this->date = $date;
        $this->storeManager = $storeManager;
    }

    public function execute(EventObserver $observer)
    {
        if (!self::$isFirstSubscribe) {
            return;
        }

        $subscriber = $observer->getSubscriber();

        if (!self::$onNewsletterSubscriberSaveAfterChecked) {
            $customer = $this->customerFactory->create();
            $websiteId = $this->storeManager->getStore($subscriber->getStoreId())->getWebsiteId();
            $customer->setWebsiteId($websiteId);
            $customer->loadByEmail($subscriber->getEmail());

            if (!$customer->getId()) {
                $customer->addData(
                    [
                        'email' => $subscriber->getSubscriberEmail(),
                        'store_id' => $subscriber->getStoreId(),
                        'group_id' => GroupManagement::NOT_LOGGED_IN_ID
                    ]
                );
            }

            if ($subscriber->getChangeStatusAt()
                && $subscriber->getSubscriberStatus() == Subscriber::STATUS_UNSUBSCRIBED
            ) {
                $subscriber->setSubscriberStatus(Subscriber::STATUS_SUBSCRIBED);
            }

            $this->scheduleFactory->create()->checkSubscriptionRules(
                $subscriber,
                $customer,
                [Rule::TYPE_CUSTOMER_SUBSCRIPTION]
            );

            self::$onNewsletterSubscriberSaveAfterChecked = true;
            $subscriber->setChangeStatusAt($this->date->date("Y-m-d H:i:s"));
            $subscriber->save();
        }
    }
}
