<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Follow Up Email for Magento 2
 */

namespace Amasty\Followup\Model\Source\Rule;

use Amasty\Followup\Model\Rule as Rule;
use Magento\Framework\Data\OptionSourceInterface;
use Magento\Sales\Model\Order as SalesOrder;
use Magento\Sales\Model\ResourceModel\Order\Status\CollectionFactory;

class CancelEventType implements OptionSourceInterface
{
    /**
     * @var CollectionFactory
     */
    private $collectionFactory;

    /**
     * @var StartEventType
     */
    private $startEventType;

    /**
     * @var array
     */
    private $state = [
        SalesOrder::STATE_PROCESSING,
        SalesOrder::STATE_COMPLETE,
        SalesOrder::STATE_CLOSED,
        SalesOrder::STATE_CANCELED
    ];

    public function __construct(
        CollectionFactory $collectionFactory,
        StartEventType $startEventType
    ) {
        $this->collectionFactory = $collectionFactory;
        $this->startEventType = $startEventType;
    }

    public function toOptionArray(): array
    {
        return [
            [
                'label' => __('Customer logged in'),
                'value' => Rule::TYPE_CANCEL_CUSTOMER_LOGGEDIN
            ],
            [
                'label' => __('New Order Placed'),
                'value' => Rule::TYPE_CANCEL_ORDER_COMPLETE
            ],
            [
                'label' => __('Customer clicked on a link in the email'),
                'value' => Rule::TYPE_CANCEL_CUSTOMER_CLICKLINK
            ],
            [
                'label' => __('Customer wishlist shared'),
                'value' => Rule::TYPE_CANCEL_CUSTOMER_WISHLIST_SHARED
            ]
        ];
    }

    public function getByType(string $type): array
    {
        $options = [];
        if ($this->startEventType->isOrderRelated($type)) {
            $orderStatusCollection = $this->collectionFactory->create();
            $orderStatus = $orderStatusCollection
                ->joinStates()
                ->addFieldToFilter('state_table.state', ['in' => $this->state])
                ->addFieldToFilter('state_table.is_default', ['eq' => 1]);

            foreach ($orderStatus as $status) {
                $options[] = [
                    'label' => __('Order Becomes: %1', $status->getLabel()),
                    'value' => Rule::TYPE_CANCEL_ORDER_STATUS . $status->getStatus()
                ];
            }
        } elseif ($this->startEventType->isWishlistRelated($type)) {
            $options[] = [
                'label' => __('Another Product Added to Wishlist'),
                'value' => Rule::TYPE_CANCEL_CUSTOMER_WISHLIST_UPDATED
            ];
        }

        return array_merge($this->toOptionArray(), $options);
    }
}
