<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Follow Up Email for Magento 2
 */

namespace Amasty\Followup\Model;

use Amasty\Followup\Api\Data\RuleInterface;
use Amasty\Followup\Model\Event\Basic;
use Amasty\Followup\Model\ResourceModel\History\CollectionFactory as HistoryCollectionFactory;
use Amasty\Followup\Model\ResourceModel\Rule\Collection as RuleCollection;
use Amasty\Followup\Model\ResourceModel\Rule\CollectionFactory as RuleCollectionFactory;
use Amasty\Followup\Model\ResourceModel\Schedule as ResourceSchedule;
use Amasty\Followup\Model\ResourceModel\Schedule\Collection as ScheduleCollection;
use Magento\Catalog\Model\Product;
use Magento\Customer\Api\Data\GroupInterface;
use Magento\Customer\Model\Customer;
use Magento\Customer\Model\Data\Customer as CustomerDataModel;
use Magento\Customer\Model\Log;
use Magento\Customer\Model\Logger;
use Magento\Customer\Model\ResourceModel\GroupRepository;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\MailException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\Context;
use Magento\Framework\Registry;
use Magento\Framework\Stdlib\DateTime;
use Magento\Newsletter\Model\Subscriber;
use Magento\Quote\Model\Quote;
use Magento\Sales\Model\Order;
use Magento\Sales\Model\Order\Address\Renderer;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\StoreManagerInterface;

class Schedule extends AbstractModel
{
    /**
     * @var array
     */
    protected $scheduleCollections = [];

    /**
     * @var array
     */
    protected $customerGroup = [];

    /**
     * @var array
     */
    protected $rules = [];

    /**
     * @var DateTime
     */
    protected $dateTime;

    /**
     * @var DateTime\DateTime
     */
    protected $date;

    /**
     * @var FlagRegistry
     */
    private $flagRegistry;

    /**
     * @var Logger
     */
    protected $logger;

    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var array
     */
    protected $customerLog = [];

    /**
     * @var HistoryCollectionFactory
     */
    protected $historyCollectionFactory;

    /**
     * @var HistoryFactory
     */
    protected $historyFactory;

    /**
     * @var GroupRepository
     */
    protected $groupRepository;

    /**
     * @var RuleCollectionFactory
     */
    protected $ruleCollectionFactory;

    /**
     * @var RuleFactory
     */
    protected $ruleFactory;

    /**
     * @var Renderer
     */
    private $addressRenderer;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    public function __construct(
        Context $context,
        Registry $registry,
        ResourceSchedule $resource,
        ScheduleCollection $resourceCollection,
        StoreManagerInterface $storeManager,
        DateTime\DateTime $date,
        DateTime $dateTime,
        FlagRegistry $flagRegistry,
        HistoryCollectionFactory $historyCollectionFactory,
        HistoryFactory $historyFactory,
        Logger $logger,
        GroupRepository $groupRepository,
        RuleCollectionFactory $ruleCollectionFactory,
        RuleFactory $ruleFactory,
        Renderer $addressRenderer,
        ConfigProvider $configProvider,
        array $data = []
    ) {
        $this->storeManager = $storeManager;
        $this->dateTime = $dateTime;
        $this->date = $date;
        $this->flagRegistry = $flagRegistry;
        $this->historyFactory = $historyFactory;
        $this->logger = $logger;
        $this->groupRepository = $groupRepository;
        $this->ruleCollectionFactory = $ruleCollectionFactory;

        parent::__construct(
            $context,
            $registry,
            $resource,
            $resourceCollection,
            $data
        );

        $this->historyCollectionFactory = $historyCollectionFactory;
        $this->ruleFactory = $ruleFactory;
        $this->addressRenderer = $addressRenderer;
        $this->configProvider = $configProvider;
    }

    public function _construct()
    {
        $this->_init(ResourceSchedule::class);
    }

    public function getConfig(): array
    {
        $config = $this->getData();
        unset($config['rule_id']);
        $config['days'] = $this->getDays();
        $config['hours'] = $this->getHours();
        $config['minutes'] = $this->getMinutes();
        $config['discount_amount'] = $config['discount_amount'] * 1;
        $config['discount_qty'] = $config['discount_qty'] * 1;

        return $config;
    }

    public function getDeliveryTime()
    {
        return ($this->getDays() * 24 * 60 * 60) +
            ($this->getHours() * 60 * 60) +
            ($this->getMinutes() * 60);
    }

    private function getScheduleCollection(Rule $rule): ScheduleCollection
    {
        if (!isset($this->scheduleCollections[$rule->getId()])) {
            $this->scheduleCollections[$rule->getId()] = $this
                ->getCollection()
                ->addRule($rule);
        }

        return $this->scheduleCollections[$rule->getId()];
    }

    /**
     * @param CustomerDataModel|Customer $customer
     * @param History $history
     * @return array
     */
    public function getCustomerEmailVars(
        $customer,
        History $history
    ): array {
        $logCustomer = $this->loadCustomerLog($customer);
        $customerGroup = $this->loadCustomerGroup((int)$customer->getGroupId());
        $customerName = $customer->getFirstname() . ' ' . $customer->getLastname();

        return [
            Formatmanager::TYPE_CUSTOMER => $customer,
            Formatmanager::TYPE_CUSTOMER_EMAIL => $customer->getEmail() ?? '',
            Formatmanager::TYPE_CUSTOMER_ID => $customer->getId(),
            Formatmanager::TYPE_CUSTOMER_GROUP => $customerGroup,
            Formatmanager::TYPE_CUSTOMER_LOG => $logCustomer,
            Formatmanager::TYPE_HISTORY => $history,
            Formatmanager::TYPE_HISTORY_ID => $history->getId(),
            Formatmanager::TYPE_CUSTOMER_NAME => $customerName,
            Formatmanager::TYPE_CUSTOMER_GROUP_CODE => $customerGroup->getCode(),
            Formatmanager::TYPE_HISTORY_COUPON_CODE => $history->getCouponCode()
        ];
    }

    /**
     * @param CustomerDataModel|Customer $customer
     * @return Log
     */
    private function loadCustomerLog($customer): Log
    {
        $customerId = $customer->getId();

        if (!isset($this->customerLog[$customerId])) {
            $this->customerLog[$customerId] = $this->logger->get($customerId);
        }

        return $this->customerLog[$customerId];
    }

    private function loadCustomerGroup(int $id): GroupInterface
    {
        if (!isset($this->customerGroup[$id])) {
            $this->customerGroup[$id] = $this->groupRepository->getById($id);
        }

        return $this->customerGroup[$id];
    }

    /**
     * @param Rule $rule
     * @param Basic $event
     * @param CustomerDataModel|Customer $customer
     * @param Product|null $product
     * @param string|null $scheduledDate
     * @return array
     * @throws LocalizedException
     * @throws MailException
     * @throws NoSuchEntityException
     */
    public function createCustomerHistory(
        Rule $rule,
        Basic $event,
        $customer,
        ?Product $product = null,
        ?string $scheduledDate = null
    ): array {
        $customerHistory = [];
        $scheduleCollection = $this->getScheduleCollection($rule);
        $scheduledAt = null;

        foreach ($scheduleCollection as $schedule) {
            $this->storeManager->setCurrentStore($customer->getStoreId());
            $history = $this->historyFactory->create();
            $history->initCustomerItem($customer);

            if ($product instanceof Product) {
                $scheduledAt = $product->getSpecialFromDate();

                /**
                 * Magento sets the DateTime object with the current date to Special Price From attribute if it's empty.
                 * @see \Magento\Catalog\Observer\SetSpecialPriceStartDate
                 */
                if ($scheduledAt instanceof \DateTime) {
                    $scheduledAt = $scheduledAt->format(DateTime::DATETIME_PHP_FORMAT);
                }

                $scheduledDate = $this->dateTime->formatDate($this->date->gmtTimestamp());
            }

            $history->createItem($schedule, $scheduledDate, $scheduledAt ?? $scheduledDate);

            if (null === $product) {
                $email = $event->getEmail($schedule, $history, $this->getCustomerEmailVars($customer, $history));
                $history->saveEmail($email);
            }

            $customerHistory[] = $history;
        }

        return $customerHistory;
    }

    /**
     * @throws LocalizedException
     * @throws MailException
     * @throws NoSuchEntityException
     */
    public function createOrderHistory(
        Rule $rule,
        Basic $event,
        Order $order,
        Quote $quote,
        CustomerDataModel $customer,
        ?string $scheduledDate = null
    ): array {
        $orderHistory = [];
        $scheduleCollection = $this->getScheduleCollection($rule);

        if (!$customer->getId()) {
            $this->initCustomer($customer, $order);
            $quote->setCustomerId(null);
        }

        foreach ($scheduleCollection as $schedule) {
            $this->storeManager->setCurrentStore($quote->getStoreId());
            $history = $this->historyFactory->create();
            $history->initOrderItem($order, $quote);
            $history->createItem($schedule, $scheduledDate ?? $order->getUpdatedAt());
            $email = $event->getEmail(
                $schedule,
                $history,
                $this->getOrderEmailVars($order, $quote, $customer, $history)
            );

            if (!$email) {
                $history->setStatus(History::STATUS_NO_PRODUCT);
                $history->save();
            } else {
                $history->saveEmail($email);
                $orderHistory[] = $history;
            }
        }

        return $orderHistory;
    }

    public function initCustomer(
        CustomerDataModel $customer,
        Order $order
    ): void {
        $data = $order->getBillingAddress()->getData();

        foreach ($data as $key => $val) {
            if ($key !== 'entity_id' && $key !== 'parent_id') {
                $customer->setData($key, $val);
            }
        }

        $customer->setData('group_id', $order->getCustomerGroupId());
    }

    private function getOrderEmailVars(
        Order $order,
        Quote $quote,
        CustomerDataModel $customer,
        History $history
    ): array {
        $vars = $this->getCustomerEmailVars($customer, $history);
        $vars[Formatmanager::TYPE_ORDER] = $order;
        $vars[Formatmanager::TYPE_ORDER_ID] = $order->getId();
        $vars[Formatmanager::TYPE_ORDER_STATUS] = $order->getStatusLabel();
        $vars[Formatmanager::TYPE_ORDER_INCREMENT_ID] = $order->getIncrementId();
        $vars[Formatmanager::TYPE_ORDER_SHIPPING_METHOD] = $order->getShippingDescription();
        $vars[Formatmanager::TYPE_QUOTE_ID] = $quote->getId();
        $vars[Formatmanager::TYPE_HISTORY_ID] = $history->getId();
        $vars[Formatmanager::TYPE_ORDER_FORMATTED_SHIPPING_ADDRESS] = $this->getFormattedShippingAddress($order);
        $vars[Formatmanager::TYPE_ORDER_FORMATTED_BILLING_ADDRESS] = $this->getFormattedBillingAddress($order);
        $vars[Formatmanager::TYPE_ORDER_PAYMENT_METHOD] = $this->configProvider->getPaymentMethodTitle(
            $order->getPayment()->getMethod(),
            (int)$quote->getStoreId()
        );
        $vars[Formatmanager::TYPE_CUSTOMER_EMAIL] = $vars[Formatmanager::TYPE_CUSTOMER_EMAIL]
            ?: $order->getCustomerEmail();

        return $vars;
    }

    private function getFormattedShippingAddress(Order $order): ?string
    {
        return !$order->getIsVirtual() && $order->getShippingAddress()
            ? $this->addressRenderer->format($order->getShippingAddress(), 'html')
            : null;
    }

    private function getFormattedBillingAddress(Order $order): ?string
    {
        return $order->getBillingAddress()
            ? $this->addressRenderer->format($order->getBillingAddress(), 'html')
            : null;
    }

    public function getDays()
    {
        return $this->getDelayedStart() > 0 && floor($this->getDelayedStart() / 24 / 60 / 60) ?
            floor($this->getDelayedStart() / 24 / 60 / 60) :
            null;
    }

    public function getHours()
    {
        $days = $this->getDays();
        $time = $this->getDelayedStart() - ($days * 24 * 60 * 60);

        return $time > 0 ?
            floor($time / 60 / 60) :
            null;
    }

    public function getMinutes()
    {
        $days = $this->getDays();
        $hours = $this->getHours();
        $time = $this->getDelayedStart() - ($days * 24 * 60 * 60) - ($hours * 60 * 60);

        return $time > 0 ?
            floor($time / 60) :
            null;
    }

    public function checkCustomerRules($customer, $types = [], $product = null)
    {
        $ruleCollection = $this->getRuleCollection($types);

        foreach ($ruleCollection as $rule) {
            $event = $rule->getStartEvent();

            if ($event->validate($customer)) {
                $this->createCustomerHistory($rule, $event, $customer, $product);
            }
        }
    }

    public function getRuleCollection(array $types = []): RuleCollection
    {
        $ruleCollection = $this->ruleCollectionFactory->create()->addStartFilter($types);

        return $ruleCollection;
    }

    public function checkSubscriptionRules(
        Subscriber $subscriber,
        Customer $customer,
        array $types = []
    ): void {
        $ruleCollection = $this->getRuleCollection($types);

        foreach ($ruleCollection as $rule) {
            $event = $rule->getStartEvent();

            if ($event->validateSubscription($subscriber, $customer)) {
                $this->createCustomerHistory($rule, $event, $customer);
            }
        }
    }

    public function process(): void
    {
        $historyCollection = $this->historyCollectionFactory->create()
            ->addOrderData()
            ->addReadyFilter(
                $this->dateTime->formatDate(
                    $this->flagRegistry->getCurrentExecution()
                )
            );

        foreach ($historyCollection as $history) {
            $rule = $this->loadRule((int)$history->getRuleId());

            if ($history->validateBeforeSent($rule)) {
                $history->processItem($rule, $history->getEmail());
            } else {
                $history->cancelItem();
            }
        }
    }

    private function loadRule(int $ruleId): RuleInterface
    {
        if (!isset($this->rules[$ruleId])) {
            $this->rules[$ruleId] = $this->ruleFactory->create()->load($ruleId);
        }

        return $this->rules[$ruleId];
    }
}
