<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Follow Up Email for Magento 2
 */

namespace Amasty\Followup\Model;

use Amasty\Followup\Api\Data\RuleInterface;
use Amasty\Followup\Api\Data\RuleInterfaceFactory;
use Amasty\Followup\Api\RuleRepositoryInterface;
use Amasty\Followup\Model\ResourceModel\Rule as RuleResourceModel;
use Amasty\Followup\Model\ResourceModel\Rule\Collection as RuleCollection;
use Amasty\Followup\Model\ResourceModel\Rule\Collection as RuleCollectionFactory;
use Amasty\Followup\Api\Data\RuleSearchResultsInterface;
use Amasty\Followup\Api\Data\RuleSearchResultsInterfaceFactory;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Api\ExtensionAttribute\JoinProcessorInterface;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Api\DataObjectHelper;
use Magento\Framework\Reflection\DataObjectProcessor;

class RuleRepository implements RuleRepositoryInterface
{
    /**
     * @var RuleResourceModel
     */
    private $resource;

    /**
     * @var RuleInterfaceFactory
     */
    private $ruleInterfaceFactory;

    /**
     * @var RuleCollectionFactory
     */
    private $ruleCollectionFactory;

    /**
     * @var JoinProcessorInterface
     */
    private $extensionAttributesJoinProcessor;

    /**
     * @var CollectionProcessorInterface
     */
    private $collectionProcessor;

    /**
     * @var RuleSearchResultsInterfaceFactory
     */
    private $searchResultsFactory;

    /**
     * @var DataObjectHelper
     */
    private $dataObjectHelper;

    /**
     * @var DataObjectProcessor
     */
    private $dataObjectProcessor;

    /**
     * @var array
     */
    private $registry = [];

    public function __construct(
        RuleResourceModel $resource,
        RuleInterfaceFactory $ruleInterfaceFactory,
        RuleCollectionFactory $ruleCollectionFactory,
        JoinProcessorInterface $extensionAttributesJoinProcessor,
        CollectionProcessorInterface $collectionProcessor,
        RuleSearchResultsInterfaceFactory $searchResultsFactory,
        DataObjectHelper $dataObjectHelper,
        DataObjectProcessor $dataObjectProcessor
    ) {
        $this->resource = $resource;
        $this->ruleInterfaceFactory = $ruleInterfaceFactory;
        $this->ruleCollectionFactory = $ruleCollectionFactory;
        $this->extensionAttributesJoinProcessor = $extensionAttributesJoinProcessor;
        $this->collectionProcessor = $collectionProcessor;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->dataObjectHelper = $dataObjectHelper;
        $this->dataObjectProcessor = $dataObjectProcessor;
    }

    public function getById(int $ruleId): RuleInterface
    {
        if (!isset($this->registry[$ruleId])) {
            /** @var RuleInterface $rule */
            $rule = $this->ruleInterfaceFactory->create();
            $this->resource->load($rule, $ruleId);
            if (!$rule->getId()) {
                throw NoSuchEntityException::singleField('id', $ruleId);
            }
            $this->registry[$ruleId] = $rule;
        }

        return $this->registry[$ruleId];
    }

    public function save(RuleInterface $rule): RuleInterface
    {
        try {
            $this->resource->save($rule);
            $this->registry[$rule->getId()] = $rule;
        } catch (\Exception $exception) {
            throw new CouldNotSaveException(__($exception->getMessage()));
        }

        return $rule;
    }

    public function delete(RuleInterface $rule): bool
    {
        try {
            $this->resource->delete($rule);
        } catch (\Exception $exception) {
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }
        if (isset($this->registry[$rule->getId()])) {
            unset($this->registry[$rule->getId()]);
        }

        return true;
    }

    public function deleteById(int $ruleId): bool
    {
        return $this->delete($this->getById($ruleId));
    }

    public function getList(SearchCriteriaInterface $searchCriteria): RuleSearchResultsInterface
    {
        /** @var RuleCollection $collection */
        $collection = $this->ruleCollectionFactory->create();

        $this->extensionAttributesJoinProcessor->process($collection, RuleInterface::class);
        $this->collectionProcessor->process($searchCriteria, $collection);

        /** @var RuleSearchResultsInterface $searchResults */
        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($searchCriteria);
        $searchResults->setTotalCount($collection->getSize());

        $objects = [];
        /** @var Rule $item */
        foreach ($collection->getItems() as $item) {
            $objects[] = $this->getDataObject($item);
        }
        $searchResults->setItems($objects);

        return $searchResults;
    }

    private function getDataObject(Rule $model): RuleInterface
    {
        /** @var RuleInterface $object */
        $object = $this->ruleInterfaceFactory->create();
        $this->dataObjectHelper->populateWithArray(
            $object,
            $this->dataObjectProcessor->buildOutputDataArray($model, RuleInterface::class),
            RuleInterface::class
        );

        return $object;
    }
}
