<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Follow Up Email for Magento 2
 */

namespace Amasty\Followup\Model;

use Amasty\Followup\Api\Data\RuleInterface as DataRuleInterface;
use Amasty\Followup\Model\ResourceModel\Rule as FollowupRuleResource;
use Amasty\Followup\Model\ResourceModel\Rule\Collection as FollowupRuleCollection;
use Amasty\Followup\Model\ResourceModel\Schedule\Collection as FollowupScheduleCollection;
use Amasty\Followup\Model\ResourceModel\Schedule\CollectionFactory as FollowupScheduleCollectionFactory;
use Amasty\Followup\Model\Source\Rule\StartEventType;
use Magento\Framework\Data\FormFactory;
use Magento\Framework\Model\Context;
use Magento\Framework\ObjectManagerInterface;
use Magento\Framework\Registry;
use Magento\Framework\Stdlib\DateTime\TimezoneInterface;
use Magento\Sales\Model\Order as SalesOrder;
use Magento\SalesRule\Model\Coupon\CodegeneratorFactory;
use Magento\SalesRule\Model\CouponFactory;
use Magento\SalesRule\Model\ResourceModel\Coupon\Collection as CouponCollection;
use Magento\SalesRule\Model\Rule as SalesRuleRule;
use Magento\SalesRule\Model\Rule\Condition\CombineFactory;
use Magento\SalesRule\Model\Rule\Condition\Product\CombineFactory as ProductCombineFactory;
use Magento\Store\Model\StoreManagerInterface;

class Rule extends SalesRuleRule implements DataRuleInterface
{
    /**
     * Rule and Store relation
     */
    public const RELATION_STORE_RULE_ID = 'rule_id';
    public const STORE_ID = 'store_id';

    /**
     * Rule and Customer group relation
     */
    public const RELATION_CUSTOMER_GROUP_RULE_ID = 'rule_id';
    public const CUSTOMER_GROUP_ID = 'customer_group_id';

    /**
     * Rule and Segment relation
     */
    public const RELATION_SEGMENT_RULE_ID = 'rule_id';
    public const SEGMENT_ID = 'segment_id';

    /**
     * Coupon code attributes
     */
    public const COUPON_CODE_NONE = 'None';
    public const COUPON_CODE_BY_PERCENT = 'by_percent';
    public const COUPON_CODE_BY_FIXED = 'by_fixed';
    public const COUPON_CODE_CART_FIXED = 'cart_fixed';
    public const COUPON_CODE_USE_RULE = 'use_rule';

    /**
     * Rule statuses
     */
    public const RULE_ACTIVE = '1';
    public const RULE_INACTIVE = '0';

    /**
     * Lists of order events
     */
    public const TYPE_ORDER_NEW = 'order_new';
    public const TYPE_ORDER_SHIP = 'order_ship';
    public const TYPE_ORDER_INVOICE = 'order_invoice';
    public const TYPE_ORDER_COMPLETE = 'order_complete';
    public const TYPE_ORDER_CANCEL = 'order_cancel';

    /**
     * Lists of customer events
     */
    public const TYPE_CUSTOMER_GROUP = 'customer_group';
    public const TYPE_CUSTOMER_BIRTHDAY = 'customer_birthday';
    public const TYPE_CUSTOMER_NEW = 'customer_new';
    public const TYPE_CUSTOMER_SUBSCRIPTION = 'customer_subscription';
    public const TYPE_CUSTOMER_ACTIVITY = 'customer_activity';
    public const TYPE_CUSTOMER_WISHLIST = 'customer_wishlist';
    public const TYPE_CUSTOMER_WISHLIST_SHARED = 'customer_wishlist_shared';
    public const TYPE_CUSTOMER_WISHLIST_SALE = 'customer_wishlist_sale';
    public const TYPE_CUSTOMER_WISHLIST_BACK_INSTOCK = 'customer_wishlist_back_instock';
    public const TYPE_CUSTOMER_DATE = 'customer_date';

    /**
     * Lists of cancel events
     */
    public const TYPE_CANCEL_ORDER_COMPLETE = 'cancel_order_complete';
    public const TYPE_CANCEL_ORDER_STATUS = 'cancel_order_status';
    public const TYPE_CANCEL_CUSTOMER_LOGGEDIN = 'cancel_customer_loggedin';
    public const TYPE_CANCEL_CUSTOMER_CLICKLINK = 'cancel_customer_clicklink';
    public const TYPE_CANCEL_CUSTOMER_WISHLIST_SHARED = 'cancel_customer_wishlist_shared';
    public const TYPE_CANCEL_CUSTOMER_WISHLIST_UPDATED = 'cancel_customer_wishlist_updated';

    /**
     * Default event
     */
    public const TYPE_BASIC = 'basic';

    /**
     * @var SalesRule
     */
    private $salesRule;

    /**
     * @var FollowupScheduleCollection
     */
    private $scheduleCollection;

    /**
     * @var ObjectManagerInterface
     */
    private $objectManager;

    /**
     * @var EventCreator
     */
    private $eventCreator;

    /**
     * @var StartEventType
     */
    private $startEventType;

    /**
     * @var SalesRuleFactory
     */
    private $salesRuleFactory;

    /**
     * @var FollowupScheduleCollectionFactory
     */
    private $scheduleCollectionFactory;

    public function __construct(
        Context $context,
        Registry $registry,
        FormFactory $formFactory,
        TimezoneInterface $localeDate,
        CouponFactory $couponFactory,
        CodegeneratorFactory $codegenFactory,
        CombineFactory $condCombineFactory,
        ProductCombineFactory $condProdCombineF,
        CouponCollection $couponCollection,
        StoreManagerInterface $storeManager,
        FollowupRuleResource $resource,
        FollowupRuleCollection $resourceCollection,
        ObjectManagerInterface $objectManager,
        EventCreator $eventCreator,
        StartEventType $startEventType,
        SalesRuleFactory $salesRuleFactory,
        FollowupScheduleCollectionFactory $scheduleCollectionFactory,
        $data = []
    ) {
        $this->objectManager = $objectManager;
        $this->eventCreator = $eventCreator;
        $this->startEventType = $startEventType;
        $this->salesRuleFactory = $salesRuleFactory;
        $this->scheduleCollectionFactory = $scheduleCollectionFactory;

        parent::__construct(
            $context,
            $registry,
            $formFactory,
            $localeDate,
            $couponFactory,
            $codegenFactory,
            $condCombineFactory,
            $condProdCombineF,
            $couponCollection,
            $storeManager,
            $resource,
            $resourceCollection,
            $data
        );
    }

    public function _construct()
    {
        $this->_init(FollowupRuleResource::class);
    }

    public function getSalesRule(): SalesRule
    {
        if (!$this->salesRule) {
            $this->salesRule = $this->salesRuleFactory->create()->load($this->getId());
        }

        return $this->salesRule;
    }

    public function getScheduleCollection(): FollowupScheduleCollection
    {
        if (!$this->scheduleCollection) {
            $this->scheduleCollection = $this->scheduleCollectionFactory->create()
                ->addFieldToFilter('rule_id', $this->getId());
        }

        return $this->scheduleCollection;
    }

    /**
     * @return Event\Basic
     */
    public function getStartEvent()
    {
        if ($this->getStartEventType()) {
            return $this->eventCreator->create($this->getStartEventType(), ['rule' => $this]);
        }

        return $this->eventCreator->create(self::TYPE_BASIC);
    }

    /**
     * @return void
     */
    public function saveSchedule()
    {
        $schedule = $this->getSchedule();

        $savedIds = [];

        if (is_array($schedule) && count($schedule) > 0) {
            foreach ($schedule as $config) {
                $object = $this->objectManager
                    ->create(\Amasty\Followup\Model\Schedule::class);

                if (isset($config['schedule_id'])) {
                    $object->load($config['schedule_id']);
                }

                $deliveryTime = $config['delivery_time'];
                $days = $deliveryTime['days'];
                $hours = $deliveryTime['hours'];
                $minutes = $deliveryTime['minutes'];
                $delayedStart = $this->_toSeconds($days, $hours, $minutes);

                $coupon = $config['coupon'];

                if (!isset($coupon[self::COUPON_CODE_USE_RULE])) {
                    $coupon[self::COUPON_CODE_USE_RULE] = false;
                }

                $coupon['rule_id'] = $this->getId();
                $coupon['email_template_id'] = $config['email_template_id'];
                $coupon['delayed_start'] = $delayedStart;

                $object->addData($coupon);
                $object->save();

                $savedIds[] = $object->getId();
            }
            $deleteCollection = $this->objectManager
                ->create(\Amasty\Followup\Model\Schedule::class)->getCollection()
                ->addFieldToFilter('rule_id', $this->getId())
                ->addFieldToFilter(
                    'schedule_id',
                    [
                        'nin' => $savedIds
                    ]
                );

            foreach ($deleteCollection as $delete) {
                $delete->delete();
            }

            $ruleProductAttributes = array_merge(
                $this->_getUsedAttributes($this->getConditionsSerialized()),
                $this->_getUsedAttributes($this->getActionsSerialized())
            );

            if (count($ruleProductAttributes)) {
                $this->getResource()->saveAttributes($this->getId(), $ruleProductAttributes);
            }
        }
    }

    /**
     * Return all product attributes used on serialized action or condition
     *
     * @param string|null $serializedString
     *
     * @return array
     */
    protected function _getUsedAttributes($serializedString)
    {
        $result = [];
        $pattern = '~s:46:"Magento\\\SalesRule\\\Model\\\Rule\\\Condition\\\Product";s:9:"attribute";s:\d+:"(.*?)"~s';
        $matches = [];

        if ($serializedString && preg_match_all($pattern, $serializedString, $matches)) {
            foreach ($matches[1] as $attributeCode) {
                $result[] = $attributeCode;
            }
        }

        return $result;
    }

    /**
     * @param string $days
     * @param string $hours
     * @param string $minutes
     *
     * @return int
     */
    protected function _toSeconds($days, $hours, $minutes)
    {
        return (int)$minutes * 60 + ((int)$hours * 60 * 60) + ((int)$days * 24 * 60 * 60);
    }

    /**
     * @param \Magento\Quote\Model\Quote $quote
     *
     * @return bool
     */
    public function validateConditions($quote)
    {
        /** @var \Magento\Quote\Model\Quote\Address $address */
        foreach ($quote->getAllAddresses() as $address) {
            $this->_initAddress($address, $quote);

            if (parent::validate($address)) {
                return true;
            }
        }

        return false;
    }

    /**
     * @param \Magento\Quote\Model\Quote\Address $address
     * @param \Magento\Quote\Model\Quote $quote
     */
    protected function _initAddress($address, $quote)
    {
        $address->setData('total_qty', $quote->getData('items_qty'));
    }

    public function getCancelEvents(): array
    {
        $cancelEvents = [];
        $cancelTypes = array_filter(explode(',', $this->getCancelEventType()));

        foreach ($cancelTypes as $cancelType) {
            if (strpos($cancelType, self::TYPE_CANCEL_ORDER_STATUS) === false) {
                $cancelEvents[] = $this->eventCreator->create($cancelType, ['rule' => $this]);
            }
        }

        if ($this->startEventType->isOrderRelated($this->getStartEventType())) {
            $state = [
                SalesOrder::STATE_PROCESSING,
                SalesOrder::STATE_COMPLETE,
                SalesOrder::STATE_CLOSED,
                SalesOrder::STATE_CANCELED
            ];

            $orderStatus = $this->objectManager
                ->get(\Magento\Sales\Model\ResourceModel\Order\Status\Collection::class)
                ->joinStates()
                ->addFieldToFilter('state_table.state', ['in' => $state])
                ->addFieldToFilter('state_table.is_default', ['eq' => 1]);

            /** @var \Magento\Sales\Model\Order\Status $status */
            foreach ($orderStatus as $status) {
                $eventKey = $this->getOrderCancelEventKey($status);

                if (in_array($eventKey, $cancelTypes)) {
                    $cancelEvents[] = $this->eventCreator->create(
                        self::TYPE_CANCEL_ORDER_STATUS,
                        ['rule' => $this, 'status' => $status]
                    );
                }
            }
        }

        return $cancelEvents;
    }

    /**
     * @param \Magento\Sales\Model\Order\Status $status
     *
     * @return string
     */
    public function getOrderCancelEventKey($status)
    {
        return self::TYPE_CANCEL_ORDER_STATUS . $status->getStatus();
    }

    public function getRuleId(): int
    {
        return (int)$this->getData(DataRuleInterface::RULE_ID);
    }

    public function setRuleId(int $id): DataRuleInterface
    {
        return $this->setData(DataRuleInterface::RULE_ID, $id);
    }

    public function getStartEventType(): string
    {
        return (string)$this->getData(DataRuleInterface::START_EVENT_TYPE);
    }

    public function setStartEventType(string $type): DataRuleInterface
    {
        return $this->setData(DataRuleInterface::START_EVENT_TYPE, $type);
    }

    public function getCancelEventType(): string
    {
        return (string)$this->getData(DataRuleInterface::CANCEL_EVENT_TYPE);
    }

    public function setCancelEventType(string $type): DataRuleInterface
    {
        return $this->setData(DataRuleInterface::CANCEL_EVENT_TYPE, $type);
    }

    public function getName(): string
    {
        return (string)$this->getData(DataRuleInterface::NAME);
    }

    public function setName(string $name): DataRuleInterface
    {
        return $this->setData(DataRuleInterface::NAME, $name);
    }

    public function isActive(): bool
    {
        return (bool)$this->getData(DataRuleInterface::IS_ACTIVE);
    }

    public function setIsActive(bool $status): DataRuleInterface
    {
        return $this->setData(DataRuleInterface::IS_ACTIVE, $status);
    }

    public function isToSubscribers(): bool
    {
        return (bool)$this->getData(DataRuleInterface::TO_SUBSCRIBERS);
    }

    public function setIsToSubscribers(bool $status): DataRuleInterface
    {
        return $this->setData(DataRuleInterface::TO_SUBSCRIBERS, $status);
    }

    public function getSenderName(): string
    {
        return (string)$this->getData(DataRuleInterface::SENDER_NAME);
    }

    public function setSenderName(string $name): DataRuleInterface
    {
        return $this->setData(DataRuleInterface::SENDER_NAME, $name);
    }

    public function getSenderEmail(): string
    {
        return (string)$this->getData(DataRuleInterface::SENDER_EMAIL);
    }

    public function setSenderEmail(string $email): DataRuleInterface
    {
        return $this->setData(DataRuleInterface::SENDER_EMAIL, $email);
    }

    public function getSenderCc(): string
    {
        return (string)$this->getData(DataRuleInterface::SENDER_CC);
    }

    public function setSenderCc(string $senderCc): DataRuleInterface
    {
        return $this->setData(DataRuleInterface::SENDER_CC, $senderCc);
    }

    public function getUtmSource(): string
    {
        return (string)$this->getData(DataRuleInterface::UTM_SOURCE);
    }

    public function setUtmSource(string $utmSource): DataRuleInterface
    {
        return $this->setData(DataRuleInterface::UTM_SOURCE, $utmSource);
    }

    public function getUtmMedium(): string
    {
        return (string)$this->getData(DataRuleInterface::UTM_MEDIUM);
    }

    public function setUtmMedium(string $utmMedium): DataRuleInterface
    {
        return $this->setData(DataRuleInterface::UTM_MEDIUM, $utmMedium);
    }

    public function getUtmTerm(): string
    {
        return (string)$this->getData(DataRuleInterface::UTM_TERM);
    }

    public function setUtmTerm(string $utmTerm): DataRuleInterface
    {
        return $this->setData(DataRuleInterface::UTM_TERM, $utmTerm);
    }

    public function getUtmContent(): string
    {
        return (string)$this->getData(DataRuleInterface::UTM_CONTENT);
    }

    public function setUtmContent(string $utmContent): DataRuleInterface
    {
        return $this->setData(DataRuleInterface::UTM_CONTENT, $utmContent);
    }

    public function getUtmCampaign(): string
    {
        return (string)$this->getData(DataRuleInterface::UTM_CAMPAIGN);
    }

    public function setUtmCampaign(string $utmCampaign): DataRuleInterface
    {
        return $this->setData(DataRuleInterface::UTM_CAMPAIGN, $utmCampaign);
    }

    public function getConditionsSerialized(): ?string
    {
        return $this->getData(DataRuleInterface::CONDITIONS_SERIALIZED);
    }

    public function setConditionsSerialized(string $conditionsSerialized): DataRuleInterface
    {
        return $this->setData(DataRuleInterface::CONDITIONS_SERIALIZED, $conditionsSerialized);
    }

    public function getDateEvent(): string
    {
        return (string)$this->getData(DataRuleInterface::CUSTOMER_DATE_EVENT);
    }

    public function setDateEvent(string $dateEvent): DataRuleInterface
    {
        return $this->setData(DataRuleInterface::CUSTOMER_DATE_EVENT, $dateEvent);
    }

    public function getCustomerGroupIds(): ?array
    {
        return $this->getData(DataRuleInterface::CUSTOMER_GROUP_IDS);
    }

    public function setCustomerGroupIds(?array $customerGroupIds): DataRuleInterface
    {
        return $this->setData(DataRuleInterface::CUSTOMER_GROUP_IDS, $customerGroupIds);
    }

    public function getSegmentIds(): ?array
    {
        return $this->getData(DataRuleInterface::SEGMENT_IDS);
    }

    public function setSegmentIds(?array $segmentIds): DataRuleInterface
    {
        return $this->setData(DataRuleInterface::SEGMENT_IDS, $segmentIds);
    }

    public function getStoreIds(): ?array
    {
        return $this->getData(DataRuleInterface::STORE_IDS);
    }

    public function setStoreIds(?array $storeIds): DataRuleInterface
    {
        return $this->setData(DataRuleInterface::STORE_IDS, $storeIds);
    }
}
