<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Follow Up Email for Magento 2
 */

namespace Amasty\Followup\Model\ResourceModel\Statistics;

use Amasty\Followup\Model\History as History;
use Amasty\Followup\Model\ResourceModel\Statistics as ResourceStatistics;
use Amasty\Followup\Model\Statistics;
use Magento\Framework\DB\Select;
use Magento\Framework\Model\ResourceModel\Db\Collection\AbstractCollection;

class Collection extends AbstractCollection
{
    public function _construct()
    {
        $this->_init(Statistics::class, ResourceStatistics::class);
    }

    public function addFieldToFilter($field, $condition = null)
    {
        if (in_array($field, ['sent', 'opened', 'clicked', 'open_rate', 'click_rate'])) {
            $field = new \Zend_Db_Expr('COALESCE(' . $field . ', 0)');
        }

        return parent::addFieldToFilter($field, $condition);
    }

    protected function _initSelect()
    {
        parent::_initSelect();
        $clickedQuery = $this->getConnection()->select()
            ->from(
                ['main_table' => $this->getTable('amasty_amfollowup_link')],
                [
                    'history_id' => 'main_table.history_id',
                    'clicked' => new \Zend_Db_Expr('COUNT(main_table.link)'),
                ]
            )->group('main_table.history_id');

        $emailStatQuery = $this->getConnection()->select()
            ->from(
                ['main_table' => $this->getTable('amasty_amfollowup_history')],
                [
                    'rule_id' => 'main_table.rule_id',
                    'sent' => new \Zend_Db_Expr('COUNT(main_table.history_id)'),
                    'opened' => new \Zend_Db_Expr(
                        'SUM(CASE WHEN main_table.opened > 0 THEN 1 ELSE 0 END)'
                    ),
                    'clicked' => new \Zend_Db_Expr(
                        'SUM(CASE WHEN clicked.clicked > 0 THEN 1 ELSE 0 END)'
                    ),
                ]
            )->joinLeft(
                ['clicked' => new \Zend_Db_Expr('(' . $clickedQuery . ')')],
                'main_table.history_id = clicked.history_id',
                []
            )->where('main_table.status = ?', History::STATUS_SENT)
            ->group('main_table.rule_id');

        $fullEmailStatQuery = $this->getConnection()->select()
            ->from(
                new \Zend_Db_Expr('(' . $emailStatQuery . ')'),
                [
                    'rule_id',
                    'sent' => new \Zend_Db_Expr('COALESCE(sent, 0)'),
                    'opened' => new \Zend_Db_Expr('COALESCE(opened, 0)'),
                    'clicked' => new \Zend_Db_Expr('COALESCE(clicked, 0)'),
                    'open_rate' => new \Zend_Db_Expr('COALESCE(opened / sent * 100, 0)'),
                    'click_rate' => new \Zend_Db_Expr('COALESCE(clicked / sent * 100, 0)')
                ]
            );

        $this->getSelect()
            ->reset(Select::COLUMNS)
            ->columns(
                [
                    'name',
                    'sent' => new \Zend_Db_Expr('COALESCE(email.sent, 0)'),
                    'opened' => new \Zend_Db_Expr('COALESCE(email.opened, 0)'),
                    'clicked' => new \Zend_Db_Expr('COALESCE(email.clicked, 0)'),
                    'open_rate' => new \Zend_Db_Expr('COALESCE(email.open_rate, 0)'),
                    'click_rate' => new \Zend_Db_Expr('COALESCE(email.click_rate, 0)')
                ]
            )->joinLeft(
                ['email' => new \Zend_Db_Expr('(' . $fullEmailStatQuery . ')')],
                'main_table.rule_id = email.rule_id',
                []
            );

        return $this;
    }
}
