<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Follow Up Email for Magento 2
 */

namespace Amasty\Followup\Model\ResourceModel\Rule\Customer;

use Amasty\Followup\Model\ResourceModel\Relation\Collection\Processor;
use Amasty\Followup\Model\ResourceModel\Relation\Collection\ProcessorFactory;
use Amasty\Followup\Model\ThirdParty\ModuleChecker;
use Magento\Customer\Model\ResourceModel\Customer\Collection as CustomerCollection;
use Magento\Eav\Model\Config as EavConfig;
use Magento\Eav\Model\EntityFactory as EavEntityFactory;
use Magento\Eav\Model\ResourceModel\Helper;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Data\Collection\Db\FetchStrategyInterface;
use Magento\Framework\Data\Collection\EntityFactory;
use Magento\Framework\DataObject\Copy\Config;
use Magento\Framework\DB\Adapter\AdapterInterface;
use Magento\Framework\Event\ManagerInterface;
use Magento\Framework\Model\ResourceModel\Db\VersionControl\Snapshot;
use Magento\Framework\Validator\UniversalFactory;
use Psr\Log\LoggerInterface;

class Collection extends CustomerCollection
{
    /**
     * @var ModuleChecker
     */
    private $moduleChecker;

    /**
     * @var Processor
     */
    private $relationProcessor;

    public function __construct(
        EntityFactory $entityFactory,
        LoggerInterface $logger,
        FetchStrategyInterface $fetchStrategy,
        ManagerInterface $eventManager,
        EavConfig $eavConfig,
        ResourceConnection $resource,
        EavEntityFactory $eavEntityFactory,
        Helper $resourceHelper,
        UniversalFactory $universalFactory,
        Snapshot $entitySnapshot,
        Config $fieldsetConfig,
        ModuleChecker $moduleChecker,
        ProcessorFactory $relationProcessorFactory,
        AdapterInterface $connection = null,
        $modelName = CustomerCollection::CUSTOMER_MODEL_NAME
    ) {
        $this->moduleChecker = $moduleChecker;
        $this->relationProcessor = $relationProcessorFactory->create(['collection' => $this]);
        parent::__construct(
            $entityFactory,
            $logger,
            $fetchStrategy,
            $eventManager,
            $eavConfig,
            $resource,
            $eavEntityFactory,
            $resourceHelper,
            $universalFactory,
            $entitySnapshot,
            $fieldsetConfig,
            $connection,
            $modelName
        );
    }

    protected function _initSelect()
    {
        parent::_initSelect();

        $this
            ->addNameToSelect()
            ->addAttributeToSelect('email')
            ->addAttributeToSelect('created_at')
            ->addAttributeToSelect('group_id')
            ->joinAttribute('billing_postcode', 'customer_address/postcode', 'default_billing', null, 'left')
            ->joinAttribute('billing_city', 'customer_address/city', 'default_billing', null, 'left')
            ->joinAttribute('billing_telephone', 'customer_address/telephone', 'default_billing', null, 'left')
            ->joinAttribute('billing_region', 'customer_address/region', 'default_billing', null, 'left')
            ->joinAttribute('billing_country_id', 'customer_address/country_id', 'default_billing', null, 'left');

        return $this;
    }

    public function addFieldToFilter($field, $condition = null)
    {
        if (in_array($field, ['segment_ids'])) {
            $this->addFilter('segment_ids', ['in' => $condition], 'public');

            return $this;
        }

        return parent::addFieldToFilter($field, $condition);
    }

    protected function _afterLoad()
    {
        if ($this->moduleChecker->isAmastySegmentEnabled()) {
            $this->relationProcessor->attachRelationDataToItem(
                'amasty_segments_index',
                'customer_id',
                'entity_id',
                'segment_ids',
                'segment_id'
            );
        }

        return parent::_afterLoad();
    }

    protected function _renderFiltersBefore()
    {
        if ($this->moduleChecker->isAmastySegmentEnabled()) {
            $this->relationProcessor->joinRelationTableByFilter(
                'segment_ids',
                'segment_id',
                'amasty_segments_index',
                'customer_id',
                'entity_id',
                'e'
            );
        }

        parent::_renderFiltersBefore();
    }
}
