<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Follow Up Email for Magento 2
 */

namespace Amasty\Followup\Model\ResourceModel;

use Amasty\Followup\Api\Data\RuleInterface;
use Amasty\Followup\Model\ResourceModel\Rule\Relation\HandlerInterface;
use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\ResourceModel\Db\Context;
use Magento\Framework\Model\ResourceModel\Db\VersionControl\AbstractDb;
use Magento\Framework\Model\ResourceModel\Db\VersionControl\RelationComposite;
use Magento\Framework\Model\ResourceModel\Db\VersionControl\Snapshot;

class Rule extends AbstractDb
{
    public const RULE_TABLE = 'amasty_amfollowup_rule';
    public const RULE_STORE_TABLE = 'amasty_amfollowup_rule_store';
    public const RULE_CUSTOMER_GROUP_TABLE = 'amasty_amfollowup_rule_customer_group';
    public const RULE_SEGMENT_TABLE = 'amasty_amfollowup_rule_segment';
    public const RULE_ATTRIBUTE_TABLE = 'amasty_amfollowup_attribute';

    /**
     * @var array
     */
    private $readHandlers;

    /**
     * @var array
     */
    private $saveHandlers;

    public function __construct(
        Context $context,
        Snapshot $entitySnapshot,
        RelationComposite $entityRelationComposite,
        $connectionName = null,
        array $readHandlers = [],
        array $saveHandlers = []
    ) {
        parent::__construct($context, $entitySnapshot, $entityRelationComposite, $connectionName);

        $this->checkHandlerInstance($readHandlers);
        $this->checkHandlerInstance($saveHandlers);
        $this->readHandlers = $readHandlers;
        $this->saveHandlers = $saveHandlers;
    }

    protected function _construct()
    {
        $this->_init(self::RULE_TABLE, RuleInterface::RULE_ID);
    }

    public function getAttributes()
    {
        $db = $this->getConnection();
        $tbl = $this->getTable(self::RULE_ATTRIBUTE_TABLE);

        $select = $db->select()->from($tbl, new \Zend_Db_Expr('DISTINCT code'));
        return $db->fetchCol($select);
    }

    public function saveAttributes($id, $attributes)
    {
        $db = $this->getConnection();
        $tbl = $this->getTable(self::RULE_ATTRIBUTE_TABLE);

        $db->delete($tbl, ['rule_id=?' => $id]);

        $data = [];
        foreach ($attributes as $code) {
            $data[] = [
                'rule_id' => $id,
                'code'    => $code,
            ];
        }
        $db->insertMultiple($tbl, $data);

        return $this;
    }

    public function load(AbstractModel $object, $value, $field = null)
    {
        parent::load($object, $value, $field);
        if ($object instanceof RuleInterface) {
            foreach ($this->readHandlers as $handler) {
                $handler->execute($object);
            }
        }

        return $this;
    }

    public function save(AbstractModel $object)
    {
        try {
            parent::save($object);
            $this->beginTransaction();

            foreach ($this->saveHandlers as $handler) {
                $handler->execute($object);
            }

            $this->commit();
        } catch (\Exception $exception) {
            $this->rollBack();
            throw $exception;
        }

        return $this;
    }

    private function checkHandlerInstance(array $handlers): void
    {
        foreach ($handlers as $handlerKey => $handler) {
            if (!$handler instanceof HandlerInterface) {
                throw new \InvalidArgumentException(
                    'The handler instance "' . $handlerKey . '" must implement '
                    . HandlerInterface::class
                );
            }
        }
    }
}
