<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Follow Up Email for Magento 2
 */

namespace Amasty\Followup\Model\ResourceModel\Relation\Collection;

use Magento\Framework\Model\AbstractModel;
use Magento\Framework\Model\ResourceModel\Db\Collection\AbstractCollection;

class Processor
{
    /**
     * @var AbstractCollection
     */
    private $collection;

    /**
     * @var string[]
     */
    private $relationTblAliases = [];

    public function __construct($collection)
    {
        $this->collection = $collection;
    }

    /**
     * @param string $relationTblName
     * @param string $relationTblLinkField
     * @param string $mainTblColumnName
     * @param string $itemNewFieldName
     * @param array|string $columnNamesFromRelationTbl
     * @return void
     */
    public function attachRelationDataToItem(
        string $relationTblName,
        string $relationTblLinkField,
        string $mainTblColumnName,
        string $itemNewFieldName,
        $columnNamesFromRelationTbl
    ):void {
        $connection = $this->collection->getConnection();

        $ids = $this->collection->getColumnValues($mainTblColumnName);
        if (count($ids)) {
            $relationTblAlias = $relationTblName . '_tbl';
            $select = $connection->select()
                ->from([$relationTblAlias => $this->collection->getTable($relationTblName)])
                ->where($relationTblAlias . '.' . $relationTblLinkField . ' IN (?)', $ids);
            $relationTblRows = $connection->fetchAll($select);

            foreach ($this->collection as $item) {
                $resultIds = $this->prepareItemData(
                    $item,
                    $mainTblColumnName,
                    $relationTblLinkField,
                    $columnNamesFromRelationTbl,
                    $relationTblRows
                );
                $item->setData($itemNewFieldName, $resultIds);
            }
        }
    }

    /**
     * @param AbstractModel $item
     * @param string $mainTblColumnName
     * @param string $relationTblLinkField
     * @param array|string $columnNamesFromRelationTbl
     * @param array $relationTblRows
     * @return array
     */
    private function prepareItemData(
        $item,
        string $mainTblColumnName,
        string $relationTblLinkField,
        string $columnNamesFromRelationTbl,
        array $relationTblRows
    ): array {
        $result = [];
        $mainTblColumnValue = $item->getData($mainTblColumnName);
        foreach ($relationTblRows as $row) {
            if ($row[$relationTblLinkField] == $mainTblColumnValue) {
                if (is_array($columnNamesFromRelationTbl)) {
                    $fieldValue = [];
                    foreach ($columnNamesFromRelationTbl as $columnNameRelationTbl) {
                        $fieldValue[$columnNameRelationTbl] = $row[$columnNameRelationTbl];
                    }
                    $result[] = $fieldValue;
                } else {
                    $result[] = $row[$columnNamesFromRelationTbl];
                }
            }
        }

        return $result;
    }

    /**
     * @param string $filterName
     * @param string $relationTblFilterFieldName
     * @param string $relationTblName
     * @param string $relationTblLinkField
     * @param string $mainTblColumnName
     * @param string $mainTblAlias
     * @return void
     */
    public function joinRelationTableByFilter(
        string $filterName,
        string $relationTblFilterFieldName,
        string $relationTblName,
        string $relationTblLinkField,
        string $mainTblColumnName,
        string $mainTblAlias = 'main_table'
    ):void {
        $relationTblAlias = $filterName . '_tbl';
        if ($this->collection->getFilter($filterName)) {
            if (!in_array($relationTblAlias, $this->relationTblAliases)) {
                $this->relationTblAliases[] = $relationTblAlias;
                $this->collection->getSelect()
                    ->joinLeft(
                        [$relationTblAlias => $this->collection->getTable($relationTblName)],
                        $mainTblAlias . '.' . $mainTblColumnName
                        . ' = ' . $relationTblAlias . '.' . $relationTblLinkField,
                        []
                    );
            }

            $this->collection->addFilterToMap($filterName, $relationTblAlias . '.' . $relationTblFilterFieldName);
        }
    }
}
