<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Follow Up Email for Magento 2
 */

namespace Amasty\Followup\Model;

use Magento\Framework\DataObject;
use Magento\Sales\Model\OrderRepository;

class Indexer extends DataObject
{
    /**
     * @var ScheduleFactory
     */
    private $scheduleFactory;

    /**
     * @var OrderRepository
     */
    private $orderRepository;

    /**
     * @var FlagRegistry
     */
    private $flagRegistry;

    public function __construct(
        ScheduleFactory $scheduleFactory,
        OrderRepository $orderRepository,
        FlagRegistry $flagRegistry,
        array $data = []
    ) {
        $this->scheduleFactory = $scheduleFactory;
        $this->orderRepository = $orderRepository;
        $this->flagRegistry = $flagRegistry;
        parent::__construct($data);
    }

    public function run(): void
    {
        $this->flagRegistry->updateFlagData(
            FlagRegistry::CURRENT_EXECUTION_PATH
        );

        $this->prepareOrderRules();
        $this->prepareCustomerRules();
        $this->scheduleFactory->create()->process();

        $this->flagRegistry->updateFlagData(
            FlagRegistry::LAST_EXECUTED_PATH,
            $this->flagRegistry->getCurrentExecution()
        );
    }

    private function prepareOrderRules(): void
    {
        $schedule = $this->scheduleFactory->create();
        $ruleCollection = $schedule->getRuleCollection(
            [
                Rule::TYPE_ORDER_NEW,
                Rule::TYPE_ORDER_SHIP,
                Rule::TYPE_ORDER_INVOICE,
                Rule::TYPE_ORDER_COMPLETE,
                Rule::TYPE_ORDER_CANCEL
            ]
        );

        foreach ($ruleCollection as $rule) {
            $event = $rule->getStartEvent();
            $quoteCollection = $event->getCollection();
            $quotes = $quoteCollection->getItems();

            foreach ($quotes as $quote) {
                if ($quote->getCustomerIsGuest() && $quote->getCustomerGroupId() === null) {
                    $quote->setCustomerGroupId(0);
                }
                if ($event->validate($quote)) {
                    $order = $this->orderRepository->get($quote->getOrderId());
                    $customer = $quote->getCustomer();
                    $schedule->createOrderHistory($rule, $event, $order, $quote, $customer);
                }
            }
        }
    }

    private function prepareCustomerRules(): void
    {
        $schedule = $this->scheduleFactory->create();
        $ruleCollection = $schedule->getRuleCollection(
            [
                Rule::TYPE_CUSTOMER_ACTIVITY,
                Rule::TYPE_CUSTOMER_BIRTHDAY,
                Rule::TYPE_CUSTOMER_DATE,
                Rule::TYPE_CUSTOMER_WISHLIST,
            ]
        );

        foreach ($ruleCollection as $rule) {
            $event = $rule->getStartEvent();
            $customerCollection = $event->getCollection();

            foreach ($customerCollection as $customer) {
                if ($event->validate($customer)) {
                    $schedule->createCustomerHistory($rule, $event, $customer);
                }
            }
        }
    }
}
