<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Follow Up Email for Magento 2
 */

namespace Amasty\Followup\Model;

use Magento\Framework\Flag;
use Magento\Framework\Flag\FlagResource;
use Magento\Framework\FlagFactory;
use Magento\Framework\Stdlib\DateTime\DateTime;

class FlagRegistry
{
    public const LAST_EXECUTED_PATH = 'amfollowup/common/last_executed';

    public const CURRENT_EXECUTION_PATH = 'amfollowup/common/current_execution';

    /**
     * @var int
     */
    private $actualGap = 3600;

    /**
     * @var DateTime
     */
    private $date;

    /**
     * @var FlagFactory
     */
    private $flagManagerFactory;

    /**
     * @var FlagResource
     */
    private $flagResource;

    public function __construct(
        DateTime $date,
        FlagFactory $flagManagerFactory,
        FlagResource $flagResource
    ) {
        $this->date = $date;
        $this->flagManagerFactory = $flagManagerFactory;
        $this->flagResource = $flagResource;
    }

    public function getLastExecuted(): int
    {
        $lastExecution = (int)$this->getFlag(self::LAST_EXECUTED_PATH)->getFlagData();

        if (!$lastExecution) {
            $lastExecution = $this->getCurrentExecution() - $this->actualGap;
            $this->updateFlagData(self::LAST_EXECUTED_PATH, $lastExecution);
        }

        return $lastExecution;
    }

    public function getCurrentExecution(): int
    {
        $currentExecution = (int)$this->getFlag(self::CURRENT_EXECUTION_PATH)->getFlagData();

        if (!$currentExecution) {
            $this->updateFlagData(self::CURRENT_EXECUTION_PATH);
        }

        return $currentExecution;
    }

    public function updateFlagData(string $flagCode, ?int $flagValue = null): void
    {
        $flag = $this->getFlag($flagCode);
        $flag->setFlagData($flagValue ?? $this->date->gmtTimestamp());
        $this->flagResource->save($flag);
    }

    private function getFlag(string $flagCode): Flag
    {
        $flag = $this->flagManagerFactory->create(['data' => ['flag_code' => $flagCode]]);
        $this->flagResource->load(
            $flag,
            $flagCode,
            'flag_code'
        );

        return $flag;
    }
}
