<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Follow Up Email for Magento 2
 */

namespace Amasty\Followup\Model\Event;

use Amasty\Followup\Helper\Data;
use Amasty\Followup\Model;
use Amasty\Followup\Model\ConfigProvider;
use Amasty\Followup\Model\Factories\SegmentFactory;
use Amasty\Followup\Model\FlagRegistry;
use Amasty\Followup\Model\Formatmanager;
use Amasty\Followup\Model\History;
use Amasty\Followup\Model\Rule;
use Amasty\Followup\Model\Urlmanager;
use Amasty\Segments\Model\SegmentRepository;
use Magento\Customer\Model\Backend\Customer as CustomerBackendModel;
use Magento\Customer\Model\Customer;
use Magento\Customer\Model\Data\Customer as CustomerDataModel;
use Magento\Customer\Model\ResourceModel\Customer\CollectionFactory as CustomerCollectionFactory;
use Magento\Email\Model\Template as EmailTemplate;
use Magento\Framework\Data\Collection\AbstractDb as AbstractDbCollection;
use Magento\Framework\DataObject;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\MailException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\ObjectManagerInterface;
use Magento\Framework\Stdlib\DateTime;
use Magento\Sales\Model\Order\Status as OrderStatus;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;

class Basic extends DataObject
{
    public const STATUS_KEY_CREATE_ORDER = 'new';

    public const STATUS_KEY_INVOICE_ORDER = 'invoice';

    public const STATUS_KEY_CANCEL_ORDER = 'cancel';

    public const STATUS_KEY_COMPLETE_ORDER = 'complete';

    public const STATUS_KEY_SHIP_ORDER = 'ship';

    /**
     * @var Model\Rule
     */
    protected $_rule;

    /**
     * @var string|null
     */
    private $lastExecuted = null;

    /**
     * @var string|null
     */
    private $currentExecution = null;

    /**
     * @var array
     */
    protected $_collections = [];

    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var Data
     */
    protected $_helper;

    /**
     * @var DateTime
     */
    protected $_dateTime;

    /**
     * @var DateTime\DateTime
     */
    protected $_date;

    /**
     * @var ConfigProvider
     */
    protected $configProvider;

    /**
     * @var \Magento\Framework\ObjectManagerInterface
     */
    protected $_objectManager;

    /**
     * @var mixed|null
     */
    protected $_status;

    /**
     * @var CustomerCollectionFactory
     */
    protected $customerCollectionFactory;

    /**
     * @var SegmentFactory
     */
    private $segmentFactory;

    /**
     * @var FlagRegistry
     */
    private $flagRegistry;

    public function __construct(
        StoreManagerInterface $storeManager,
        Rule $rule,
        Data $helper,
        ConfigProvider $configProvider,
        DateTime\DateTime $date,
        DateTime $dateTime,
        ObjectManagerInterface $objectManager,
        CustomerCollectionFactory $collectionCustomerFactory,
        SegmentFactory $segmentFactory,
        FlagRegistry $flagRegistry,
        ?OrderStatus $status = null,
        array $data = []
    ) {
        $this->_data = $data;
        $this->storeManager = $storeManager;
        $this->_rule = $rule;
        $this->_helper = $helper;
        $this->configProvider = $configProvider;
        $this->_dateTime = $dateTime;
        $this->_date = $date;
        $this->_objectManager = $objectManager;
        $this->_status = $status;
        $this->customerCollectionFactory = $collectionCustomerFactory;
        $this->segmentFactory = $segmentFactory;
        $this->flagRegistry = $flagRegistry;
        parent::__construct($data);
    }

    /**
     * @param int $storeId
     *
     * @return bool
     */
    protected function _validateStore($storeId)
    {
        $arrStores = $this->_rule->getStoreIds();

        return empty($arrStores) || in_array($storeId, $arrStores);
    }

    /**
     * @param string $customerEmail
     *
     * @return bool
     */
    protected function _validateCustomer($customerEmail)
    {
        $ret = true;
        $arrSegments = $this->_rule->getSegmentIds();

        if (($this->_helper->isModuleOutputEnabled(Data::AMASTY_SEGMENT_MODULE_DEPEND_NAMESPACE)
            && !empty($arrSegments))
        ) {
            /** @var SegmentRepository $segmentsRepository */
            $segmentsRepository = $this->segmentFactory->getSegmentRepository();

            foreach ($arrSegments as $segment) {
                $model = $segmentsRepository->get($segment);
                $salesRule = $model->getSalesRule();

                if ($model && $model->getId()) {
                    $customer = $this
                        ->customerCollectionFactory
                        ->create()
                        ->addFieldToFilter('email', ['eq' => $customerEmail])
                        ->getFirstItem();

                    if ($customer->getId()) {
                        $validateByIndex = $salesRule->validateByIndex(
                            $this->segmentFactory->getValidationField(),
                            $arrSegments,
                            $customer->getId()
                        );

                        if (!$validateByIndex && !$salesRule->validate($customer)) {
                            return false;
                        }
                    }
                }
            }
        }
        return $ret;
    }

    /**
     * @param int $customerGroupId
     *
     * @return bool
     */
    protected function _validateCustomerGroup($customerGroupId)
    {
        $arrCustomerGroups = $this->_rule->getCustomerGroupIds();

        return empty($arrCustomerGroups) || in_array($customerGroupId, $arrCustomerGroups);
    }

    /**
     * @param int $storeId
     * @param string $customerEmail
     * @param int $customerGroupId
     *
     * @return bool
     */
    protected function _validateBasic($storeId, $customerEmail, $customerGroupId)
    {
        return $this->_rule->getIsActive() == 1
            && $this->_validateStore($storeId)
            && $this->_validateCustomer($customerEmail)
            && $this->_validateCustomerGroup($customerGroupId);
    }

    /**
     * @param $object
     *
     * @return bool
     */
    public function validate($object)
    {
        return true;
    }

    /**
     * @return string
     */
    protected function _getCollectionKey()
    {
        return $this->_rule->getId() . "_" . $this->_rule->getStartEventType();
    }

    public function getCollection(): AbstractDbCollection
    {
        if (!isset($this->_collections[$this->_getCollectionKey()])) {
            $this->setCollection();
        }

        return $this->_collections[$this->_getCollectionKey()];
    }

    public function setCollection()
    {
        $this->_collections[$this->_getCollectionKey()] = $this->_initCollection();
    }

    protected function _initCollection()
    {
        return null;
    }

    public function clear(): void
    {
        $this->lastExecuted = null;
        $this->currentExecution = null;
        $this->_collections = [];
    }

    public function getLastExecuted(): int
    {
        if (null === $this->lastExecuted) {
            $this->lastExecuted = $this->flagRegistry->getLastExecuted();
        }

        return $this->lastExecuted;
    }

    public function getCurrentExecution(): int
    {
        if (null === $this->currentExecution) {
            $this->currentExecution = $this->flagRegistry->getCurrentExecution();
        }

        return $this->currentExecution;
    }

    /**
     * @throws MailException
     * @throws LocalizedException
     * @throws NoSuchEntityException
     */
    public function getEmail(Model\Schedule $schedule, History $history, array $vars = []): array
    {
        $templateId = $schedule->getEmailTemplateId();

        $ret = [
            'body' => '',
            'subject' => ''
        ];

        $storeId = $history->getStoreId() ?? Store::DEFAULT_STORE_ID;
        $urlManager = $this->_objectManager->create(Urlmanager::class)->init($history);
        $formatManager = $this->_objectManager->create(Formatmanager::class)->init($vars);

        $variables = array_merge(
            [
                'urlmanager' => $urlManager,
                'unsubscribeUrl' => $urlManager->unsubscribeUrl(),
                'cartUrl' => $urlManager->mageUrl('checkout/cart/index'),
                'formatmanager' => $formatManager,
                'dateOfBirth' => $formatManager->formatDate('customer', 'dob'),
                'registrationDate' => $formatManager->formatDate('customer', 'created_at'),
                'lastLoginDate' => $formatManager->formatDate('customer_log', 'login_at'),
                'lastLoginAt' => $formatManager->formatDate('customer_log', 'last_login_at'),
                'grandTotal' => $formatManager->formatPrice('order', 'grand_total'),
                'orderCreatedAt' => $formatManager->formatDate('order', 'created_at'),
                'store' => $this->storeManager->getStore($storeId),
                'store_name' => $this->storeManager->getStore($storeId)->getName()
            ],
            $vars
        );

        $emailTemplate = $this->_objectManager
            ->create(EmailTemplate::class);
        $emailTemplate->setDesignConfig(
            [
                'area' => 'frontend',
                'store' => $storeId
            ]
        );

        if (is_numeric($templateId)) {
            $emailTemplate->load($templateId);
        } else {
            $localeCode = $this->_helper->getScopeValue('general/locale/code', $storeId);
            $emailTemplate->loadDefault($templateId, $localeCode);
        }

        if (!$emailTemplate->getId()) {
            throw new MailException(
                __('Invalid transactional email code: ' . $templateId)
            );
        }

        $emailTemplate->setData('is_legacy', true);
        $ret['body'] = $emailTemplate->getProcessedTemplate($variables, true);
        $ret['subject'] = $emailTemplate->getProcessedTemplateSubject($variables);

        return $ret;
    }

    /**
     * @param CustomerDataModel|Customer $customer
     * @return int
     */
    protected function getStoreIdFromCustomer($customer): int
    {
        if ($customer instanceof CustomerBackendModel) {
            $storeId = $customer->getData('store_id');
        } else {
            $storeId = $customer->getStoreId();
        }

        return (int)$storeId;
    }
}
