<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Follow Up Email for Magento 2
 */

namespace Amasty\Followup\Controller\Adminhtml\Rule;

use Amasty\Followup\Model\ConfigProvider;
use Amasty\Followup\Model\Rule;
use Amasty\Followup\Model\RuleFactory;
use Amasty\Followup\Model\SalesRuleFactory;
use Amasty\Followup\Model\ScheduleFactory;
use Amasty\Followup\Model\Source\Rule\StartEventType;
use Magento\Backend\App\Action\Context;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Framework\Controller\AbstractResult;
use Magento\Framework\Controller\Result\Json;
use Magento\Framework\Controller\Result\JsonFactory;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\MailException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Registry;
use Magento\Framework\Stdlib\DateTime\DateTime;
use Magento\Quote\Model\QuoteRepository;
use Magento\Sales\Model\OrderRepository;

class TestEmail extends \Amasty\Followup\Controller\Adminhtml\Rule
{
    /**
     * @var RuleFactory
     */
    private $ruleFactory;

    /**
     * @var CustomerRepositoryInterface
     */
    private $customerRepository;

    /**
     * @var OrderRepository
     */
    private $orderRepository;

    /**
     * @var QuoteRepository
     */
    private $quoteRepository;

    /**
     * @var ScheduleFactory
     */
    private $scheduleFactory;

    /**
     * @var JsonFactory
     */
    private $resultJsonFactory;

    /**
     * @var DateTime
     */
    private $date;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var StartEventType
     */
    private $startEventType;

    public function __construct(
        Context $context,
        Registry $coreRegistry,
        SalesRuleFactory $salesRuleFactory,
        RuleFactory $ruleFactory,
        ScheduleFactory $scheduleFactory,
        CustomerRepositoryInterface $customerRepository,
        OrderRepository $orderRepository,
        QuoteRepository $quoteRepository,
        JsonFactory $resultJsonFactory,
        DateTime $date,
        ConfigProvider $configProvider,
        StartEventType $startEventType
    ) {
        $this->ruleFactory = $ruleFactory;
        $this->scheduleFactory = $scheduleFactory;
        $this->customerRepository = $customerRepository;
        $this->orderRepository = $orderRepository;
        $this->quoteRepository = $quoteRepository;
        $this->resultJsonFactory = $resultJsonFactory;
        $this->date = $date;
        $this->configProvider = $configProvider;
        $this->startEventType = $startEventType;

        parent::__construct(
            $context,
            $coreRegistry,
            $salesRuleFactory
        );
    }

    public function execute(): AbstractResult
    {
        /** @var Json $result */
        $resultJson = $this->resultJsonFactory->create();
        $ruleId = $this->getRequest()->getParam('rule_id');
        $rule = $this->ruleFactory->create()->load($ruleId);

        if ($this->startEventType->isOrderRelated($rule->getStartEventType())) {
            $this->testOrderRule($rule);
        } else {
            $this->testCustomerRule($rule);
        }

        $messages = $this->getMessageManager()->getMessages(true);

        return $resultJson->setData(
            [
                'error' => $messages->getCount() > 0,
                'errorMsg' => $messages->getCount() > 0 ? $messages->getLastAddedMessage()->getText() : null
            ]
        );
    }

    /**
     * @throws LocalizedException
     * @throws MailException
     * @throws NoSuchEntityException
     */
    private function testCustomerRule(Rule $rule): void
    {
        $customerId = $this->getRequest()->getParam('id');
        $customer = $this->customerRepository->getById($customerId);

        if ($rule->getId() && $customer->getId()) {
            $scheduledDate = $this->date->date();
            $schedule = $this->scheduleFactory->create();
            $event = $rule->getStartEvent();
            $isTestMode = $this->isTestMode();
            $historyItems = [];

            if ($isTestMode || $event->validate($customer)) {
                $historyItems = $schedule->createCustomerHistory(
                    $rule,
                    $event,
                    $customer,
                    null,
                    $scheduledDate
                );
            }

            if ($isTestMode && $historyItems) {
                foreach ($historyItems as $history) {
                    $history->processItem($rule, null, true);
                }
            }
        }
    }

    /**
     * @throws InputException
     * @throws NoSuchEntityException
     * @throws LocalizedException
     * @throws MailException
     */
    private function testOrderRule(Rule $rule): void
    {
        $orderId = $this->getRequest()->getParam('id');
        $order = $this->orderRepository->get($orderId);
        $quote = $this->quoteRepository->get($order->getQuoteId());
        $customer = $quote->getCustomer();

        if ($rule->getId() && $order->getId() && $quote->getId()) {
            $isTestMode = $this->isTestMode();
            $scheduledDate = $this->date->date();
            $schedule = $this->scheduleFactory->create();
            $event = $rule->getStartEvent();
            $historyItems = [];

            if ($isTestMode || $event->validate($quote)) {
                $historyItems = $schedule->createOrderHistory(
                    $rule,
                    $event,
                    $order,
                    $quote,
                    $customer,
                    $scheduledDate
                );
            }

            if ($isTestMode && $historyItems) {
                foreach ($historyItems as $history) {
                    $history->processItem($rule, null, true);
                }
            }
        }
    }

    private function isTestMode(): bool
    {
        return (bool)$this->configProvider->getRecipientEmailForTest();
    }
}
