<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Follow Up Email for Magento 2
 */
namespace Amasty\Followup\Controller\Adminhtml\Rule;

use Amasty\Base\Model\Serializer;
use Amasty\Followup\Model\RuleFactory;
use Amasty\Followup\Model\RuleRepository;
use Amasty\Followup\Model\SalesRuleFactory;
use Amasty\Followup\Model\Source\Rule\EmailTemplates;
use Magento\Backend\App\Action;
use Magento\Framework\App\Request\DataPersistorInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Registry;
use Magento\Framework\Stdlib\DateTime\DateTime;
use Psr\Log\LoggerInterface;

class Save extends \Amasty\Followup\Controller\Adminhtml\Rule
{
    public const DATA_PERSISTOR_KEY = 'amasty_followup_rule';

    /**
     * @var DateTime
     */
    protected $date;

    /**
     * @var Serializer
     */
    private $serializer;

    /**
     * @var EmailTemplates
     */
    private $emailTemplatesSource;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var RuleFactory
     */
    private $ruleFactory;

    /**
     * @var RuleRepository
     */
    private $ruleRepository;

    /**
     * @var DataPersistorInterface
     */
    private $dataPersistor;

    public function __construct(
        Action\Context $context,
        Registry $coreRegistry,
        DateTime $date,
        Serializer $serializer,
        LoggerInterface $logger,
        RuleFactory $ruleFactory,
        SalesRuleFactory $salesRuleFactory,
        EmailTemplates $emailTemplatesSource,
        RuleRepository $ruleRepository,
        DataPersistorInterface $dataPersistor
    ) {
        $this->date = $date;
        $this->serializer = $serializer;
        $this->emailTemplatesSource = $emailTemplatesSource;
        $this->logger = $logger;
        $this->ruleFactory = $ruleFactory;
        $this->ruleRepository = $ruleRepository;
        $this->dataPersistor = $dataPersistor;

        parent::__construct(
            $context,
            $coreRegistry,
            $salesRuleFactory
        );
    }

    public function execute()
    {
        if ($this->getRequest()->getPostValue()) {
            $data = $this->getRequest()->getPostValue();

            try {
                $model = $this->ruleFactory->create();
                $id = (int)$this->getRequest()->getParam('rule_id');

                if ($id) {
                    $model = $this->ruleRepository->getById($id);
                } else {
                    $data['schedule'] = [$this->getDefaultScheduleValue($data)];
                }

                if (isset($data['rule']) && isset($data['rule']['conditions'])) {
                    $data['conditions'] = $data['rule']['conditions'];
                    unset($data['rule']);
                    $salesRule = $this->salesRuleFactory->create();
                    $salesRule->loadPost($data);
                    $data['conditions_serialized'] = $this->serializer->serialize(
                        $salesRule->getConditions()->asArray()
                    );
                    unset($data['conditions']);
                }

                if (isset($data['customer_date_event'])) {
                    $data['customer_date_event'] = $this->date->date("Y-m-d H:i:s", $data['customer_date_event']);
                }

                if (isset($data['store_ids']) && !is_array($data['store_ids'])) {
                    $data['store_ids'] = null;
                }

                if (isset($data['customer_group_ids']) && !is_array($data['customer_group_ids'])) {
                    $data['customer_group_ids'] = null;
                }

                if (isset($data['segment_ids']) && !is_array($data['segment_ids'])) {
                    $data['segment_ids'] = null;
                }

                $model->setData($data);
                $this->prepareForSave($model);
                $this->dataPersistor->set(self::DATA_PERSISTOR_KEY, $model->getData());

                $this->validate($data);

                $model->save();
                $id = $model->getRuleId();

                if (isset($data['schedule'])) {
                    foreach ($data['schedule'] as &$item) {
                        $this->prepareSchedule($item);
                    }
                    $model->setSchedule($data['schedule']);
                }

                if ($model->getSchedule()) {
                    $model->saveSchedule();
                } elseif ($id) {
                    $this->messageManager->addWarningMessage(
                        __('Please set Schedule.')
                    );
                    $this->dataPersistor->set(self::DATA_PERSISTOR_KEY, $data);

                    return $this->resultRedirectFactory->create()->setPath(
                        'amasty_followup/*/edit',
                        ['id' => $model->getId()]
                    );
                }

                $this->messageManager->addSuccess(__('You saved the rule.'));
                $this->dataPersistor->clear(self::DATA_PERSISTOR_KEY);

                if ($this->getRequest()->getParam('back')) {
                    return $this->resultRedirectFactory->create()->setPath(
                        'amasty_followup/*/edit',
                        ['id' => $model->getId()]
                    );
                }

                return $this->resultRedirectFactory->create()->setPath('amasty_followup/*/');
            } catch (\Magento\Framework\Exception\LocalizedException $e) {
                $this->messageManager->addError($e->getMessage());
                $id = (int)$this->getRequest()->getParam('rule_id');

                $resultRedirect = $this->resultRedirectFactory->create();
                if (!empty($id)) {
                    $resultRedirect->setPath('amasty_followup/*/edit', ['id' => $id]);
                } else {
                    $resultRedirect->setPath('amasty_followup/*/new');
                }

                return $resultRedirect;
            } catch (\Exception $e) {
                $this->messageManager->addError(
                    __('Something went wrong while saving the rule data. Please review the error log.')
                );
                $this->logger->critical($e);
                $this->dataPersistor->set(self::DATA_PERSISTOR_KEY, $data);

                return $this->resultRedirectFactory->create()->setPath(
                    'amasty_followup/*/edit',
                    ['id' => $this->getRequest()->getParam('rule_id')]
                );
            }
        }

        return $this->resultRedirectFactory->create()->setPath('amasty_followup/*/');
    }

    public function prepareForSave($model)
    {
        $fields = ['methods', 'cancel_event_type'];

        foreach ($fields as $f) {
            // convert data from array to string
            $val = $model->getData($f);
            $model->setData($f, '');

            if (is_array($val)) {
                // need commas to simplify sql query
                $model->setData($f, ',' . implode(',', $val) . ',');
            }
        }

        return true;
    }

    /**
     * @param array $data
     * @return array
     */
    private function getDefaultScheduleValue($data)
    {
        $emailTemplateId = isset($data['start_event_type'])
            ? $this->emailTemplatesSource->getByType($data['start_event_type'])[0]['value'] : '';

        return [
            'schedule_id' => '',
            'email_template_id' => $emailTemplateId,
            'delivery_time' => [
                'days' => '',
                'hours' => '',
                'minutes' => '5'
            ],
            'coupon' => [
            ]
        ];
    }

    private function prepareSchedule(&$data)
    {
        $data['delivery_time'] = [
            'days' => '',
            'hours' => '',
            'minutes' => ''
        ];

        if (isset($data['use_rule']) && $data['use_rule'] == 1) {
            $data['coupon'] = [
                'use_rule' => '',
                'sales_rule_id' => ''
            ];
        } else {
            $data['coupon'] = [
                'coupon_type' => '',
                'discount_amount' => '',
                'expired_in_days' => '',
                'discount_qty' => '',
                'discount_step' => '',
                'use_rule' => '',
                'sales_rule_id' => ''
            ];
        }

        foreach (['delivery_time', 'coupon'] as $field) {
            foreach ($data[$field] as $key => &$value) {
                if (isset($data[$key])) {
                    $value = $data[$key];
                    unset($data[$key]);
                }
            }
        }
    }

    private function validate($data)
    {
        if (isset($data['schedule'])) {
            foreach ($data['schedule'] as $schedule) {
                if (empty($schedule['sales_rule_id'])
                    && isset($schedule['use_rule'])
                    && $schedule['use_rule'] == 1
                ) {
                    throw new LocalizedException(
                        __('Shopping Cart Rule is required')
                    );
                }
            }
        }
    }
}
