<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Follow Up Email for Magento 2
 */

namespace Amasty\Followup\Block\Email;

use Amasty\Followup\Model\History;
use Amasty\Followup\Model\HistoryFactory;
use Amasty\Followup\Model\ResourceModel\History as HistoryResource;
use Amasty\Followup\Model\Schedule;
use Amasty\Followup\Model\Urlmanager;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Helper\Image;
use Magento\Framework\ObjectManagerInterface;
use Magento\Framework\Pricing\PriceCurrencyInterface;
use Magento\Framework\Stdlib\DateTime;
use Magento\Framework\View\Element\Template;
use Magento\Framework\View\Element\Template\Context;

class Items extends Template
{
    /**
     * @var array
     */
    protected $_params = [
        'mode' => [
            'default' => 'table',
            'available' => [
                'list', 'table'
            ]
        ],
        'image' => [
            'default' => 'yes',
            'available' => [
                'yes', 'no'
            ]
        ],
        'price' => [
            'default' => 'yes',
            'available' => [
                'yes', 'no'
            ]
        ],
        'priceFormat' => [
            'default' => 'exculdeTax',
            'available' => [
                'exculdeTax', 'includeTax'
            ]
        ],
        'descriptionFormat' => [
            'default' => 'short',
            'available' => [
                'short', 'full', 'no'
            ]
        ],
        'discount' => [
            'default' => 'yes',
            'available' => [
                'yes', 'no'
            ]
        ],
        'showSpecialPrice' => [
            'default' => 'no',
            'available' => [
                'yes', 'no'
            ]
        ],
    ];

    /**
     * @var PriceCurrencyInterface
     */
    protected $priceCurrency;

    /**
     * @var ObjectManagerInterface
     */
    protected $objectManager;

    /**
     * @var Urlmanager
     */
    protected $_urlManager;

    /**
     * @var Image
     */
    protected $_imageHelper;

    /**
     * @var DateTime
     */
    protected $dateTime;

    /**
     * @var ProductRepositoryInterface
     */
    private $productRepository;

    /**
     * @var HistoryFactory
     */
    private $historyFactory;

    /**
     * @var HistoryResource
     */
    private $historyResource;

    public function __construct(
        Context $context,
        PriceCurrencyInterface $priceCurrency,
        ObjectManagerInterface $objectManager,
        Urlmanager $urlManager,
        Image $imageHelper,
        DateTime $dateTime,
        ProductRepositoryInterface $productRepository,
        HistoryFactory $historyFactory,
        HistoryResource $historyResource,
        array $data = []
    ) {
        parent::__construct($context, $data);
        $this->priceCurrency = $priceCurrency;
        $this->objectManager = $objectManager;
        $this->_urlManager = $urlManager;
        $this->_imageHelper = $imageHelper;
        $this->dateTime = $dateTime;
        $this->productRepository = $productRepository;
        $this->historyFactory = $historyFactory;
        $this->historyResource = $historyResource;
    }

    public function setMode($mode)
    {
        $this->setTemplate('email/' . $mode . '.phtml');
        return parent::setMode($mode);
    }

    public function getMode()
    {
        return $this->_getLayoutParam('mode');
    }

    public function getItems()
    {
        $items = [];
        if ($this->getQuoteId() || $this->getCustomerId()) {
            $childBlock = $this->getChildBlock('amfollowup.items.data');
            $childBlock->setQuoteId($this->getQuoteId());
            $childBlock->setCustomerId($this->getCustomerId());
            $items = $childBlock->getItems();
        }

        return $items;
    }

    public function getProductPrice($product)
    {
        if ($this->showPriceIncTax()) {
            $price = $product->getPriceInclTax();
        } else {
            $price = $product->getPrice();
            if (!$price) {
                $price = $product->getFinalPrice();
            }
        }

        return $price;
    }

    public function getFormatedProductPrice($price)
    {
        return $this->formatPrice($price);
    }

    public function formatPrice($price)
    {
        $storeId = null;

        if ($this->getQuote()) {
            $storeId = $this->getQuote()->getStore();
        } elseif ($this->getCustomer() && $this->getCustomer()->getStoreId()) {
            $storeId = $this->getCustomer()->getStoreId();
        }

        return $this->priceCurrency->format(
            $price,
            true,
            PriceCurrencyInterface::DEFAULT_PRECISION,
            $storeId
        );
    }

    public function showPriceIncTax()
    {
        return $this->_getLayoutParam('priceFormat') == 'includeTax';
    }

    public function showDiscount()
    {
        return $this->_getLayoutParam('discount') == 'yes';
    }

    public function showImage()
    {
        return $this->_getLayoutParam('image') == 'yes';
    }

    public function showSpecialPrice()
    {
        return $this->_getLayoutParam('showSpecialPrice') == 'yes';
    }

    protected function _getLayoutParam($key)
    {
        $func = 'get' . $key;
        $layoutParam = $this->$func();

        return in_array($layoutParam, $this->_params[$key]['available'])
            ? $layoutParam
            : $this->_params[$key]['default'];
    }

    public function getDiscountPrice($price)
    {
        /** @var History $history */
        $history = $this->historyFactory->create();
        $this->historyResource->load($history, $this->getHistoryId());
        $discountPrice = $price;

        $sceduleId = $history->getScheduleId();
        $schedule = $this->objectManager->create(
            Schedule::class
        );
        $schedule->load($sceduleId);

        if ($schedule->getDiscountAmount()) {
            switch ($schedule->getCouponType()) {
                case "by_percent":
                    $discountPrice -= $discountPrice * $schedule->getDiscountAmount() / 100;
                    break;
                case "by_fixed":
                    $discountPrice -= $schedule->getDiscountAmount();
                    break;
            }
        }

        return $discountPrice;
    }

    public function getProductUrl($item)
    {
        $this->_initUrlManager();

        if ($item->getRedirectUrl()) {
            return $item->getRedirectUrl();
        }

        $product = $item;

        $option = $item->getOptionByCode('product_type');
        if ($option) {
            $product = $option->getProduct();
        }

        return $this->_urlManager->get($product->getUrlModel()->getUrl($product));
    }

    protected function _initUrlManager()
    {
        /** @var History $history */
        $history = $this->historyFactory->create();
        $this->historyResource->load($history, $this->getHistoryId());

        if (!$this->_urlManager->getRule()) {
            $this->_urlManager->init($history);
        }
    }

    public function getProductImageHelper()
    {
        return $this->_imageHelper;
    }

    public function initProductImageHelper($visibleItem, $imageId)
    {
        $product = $visibleItem;

        if ($this->getQuote()) {
            foreach ($this->getQuote()->getAllItems() as $item) {
                if ($item->getParentItemId() && $item->getParentItemId() == $visibleItem->getId()) {
                    $product = $item;
                    break;
                }
            }
        }

        $this->_imageHelper->init($product, $imageId);
    }

    public function getDescription($product)
    {
        $desc = '';

        if (!$this->hideDescription()) {
            $desc = $this->showShortDescription() ? $product->getShortDescription() : $product->getDescription();
        }

        return $desc;
    }

    public function hideDescription()
    {
        return $this->_getLayoutParam('descriptionFormat') == 'no';
    }

    public function showShortDescription()
    {
        return $this->_getLayoutParam('descriptionFormat') == 'short';
    }

    public function showPrice()
    {
        return $this->_getLayoutParam('price') == 'yes';
    }

    /**
     * @param $item
     * @return \Magento\Catalog\Api\Data\ProductInterface|null
     */
    public function loadProduct($item)
    {
        $product = null;

        if ($item instanceof \Magento\Catalog\Model\Product) {
            $product = $this->productRepository->getById($item->getId());
        } elseif ($item->getQuote()) {
            $product = $this->productRepository->getById($item->getProductId(), false, $item->getQuote()->getStoreId());
        } else {
            $product = $item->getProduct();
        }

        return $product;
    }

    public function getDate($date, $includeTime = false)
    {
        return $this->dateTime->formatDate($date, $includeTime);
    }

    public function displayFormatQty($qty = 0)
    {
        return ($qty) ? '&nbsp;x&nbsp;' . $qty : '';
    }
}
