<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Duplicate Categories for Magento 2
*/

declare(strict_types=1);

namespace Amasty\DuplicateCategories\Model;

use Magento\Catalog\Model\Category;
use Magento\Catalog\Model\CategoryFactory;
use Magento\Catalog\Model\CategoryRepository;
use Magento\Catalog\Model\ResourceModel\Category\Tree;
use Magento\Framework\App\ProductMetadataInterface;
use Magento\Framework\App\Request\Http;
use Magento\Framework\Message\ManagerInterface;
use Magento\Framework\ObjectManagerInterface;
use Magento\Store\Model\StoreManagerInterface;

class CategoriesSave
{
    /**
     * @var array
     */
    private $unsetArray = [
        'is_anchor',
        'path',
        'position',
        'url_path',
        'level',
        'entity_id',
        'parent_id',
        'created_at',
        'updated_at',
        'custom_layout_update',
        'custom_layout_update_file'
    ];

    /**
     * @var bool
     */
    private $savedParentCategoryRule = false;

    /**
     * @var Tree
     */
    private $categoryTree;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var ObjectManagerInterface
     */
    private $objectManager;

    /**
     * @var Http
     */
    private $request;

    /**
     * @var ProductMetadataInterface
     */
    private $metadata;

    /**
     * @var array
     */
    private $notForCopy = [];

    /**
     * @var CategoryFactory
     */
    private $categoryFactory;

    /**
     * @var CategoryRepository
     */
    private $categoryRepository;

    /**
     * @var ManagerInterface
     */
    private $messageManager;

    public function __construct(
        Tree $categoryTree,
        StoreManagerInterface $storeManager,
        ObjectManagerInterface $objectManager,
        Http $request,
        ProductMetadataInterface $metadata,
        CategoryFactory $categoryFactory,
        CategoryRepository $categoryRepository,
        ManagerInterface $messageManager
    ) {
        $this->categoryTree = $categoryTree;
        $this->storeManager = $storeManager;
        $this->objectManager = $objectManager;
        $this->request = $request;
        $this->metadata = $metadata;
        $this->categoryFactory = $categoryFactory;
        $this->categoryRepository = $categoryRepository;
        $this->messageManager = $messageManager;
    }

    public function handleSubcategories(int $fromCategoryId, int $toCategoryId): void
    {
        if ($this->request->getParam('include_subcats')) {
            $tree = $this->categoryTree->load();
            $node = $tree->getNodeById($fromCategoryId);

            if ($childNodes = $node->getAllChildNodes()) {
                $childNodes = $this->removeChildChild($childNodes, $fromCategoryId);
                //for necessary order of elements
                $childNodes = array_reverse($childNodes);

                foreach ($childNodes as $subcategory) {
                    $fromSubCategoryId = (int)$subcategory->getId();
                    $toSubCategoryId = $this->duplicateCategory($fromSubCategoryId, $toCategoryId);
                    $subNode = $tree->getNodeById($fromSubCategoryId);

                    if ($node->getAllChildNodes()) {
                        $this->handleSubcategories((int)$subNode->getId(), $toSubCategoryId);
                    }
                }
            }
        }
    }

    private function removeChildChild($childNodes, $fromCategoryId)
    {
        foreach ($childNodes as $key => $child) {
            if ($child->getParentId() != $fromCategoryId) {
                unset($childNodes[$key]);
            }
        }

        return $childNodes;
    }

    public function duplicateCategory(int $fromCategoryId, int $toParentId): int
    {
        //fix for unnecessary duplicating one child if ($fromCategoryId == $toParentId) part #1
        if (in_array($fromCategoryId, $this->notForCopy)) {
            return $fromCategoryId;
        }

        /** @var Category $fromCategory */
        $fromCategory = $this->categoryRepository->get($fromCategoryId);

        /** @var Category $toCategory */
        $toCategory = $this->categoryFactory->create();
        $toCategory->setData($fromCategory->getData());
        $toCategory->unsetData('custom_layout_update');
        $toCategory->unsetData('custom_layout_update_file');
        $this->searchAndReplace($toCategory);
        $toCategory->setId(null);

        /** @var Category $toParentCategory */
        $toParentCategory = $this->categoryRepository->get($toParentId);

        $toCategory->setPath($toParentCategory->getPath());
        $toCategory->setParentId($toParentCategory->getId());
        //+100 - fix for not changing positions of original categories
        $toCategory->setPosition($toCategory->getPosition() + 100);
        $this->save($toCategory);

        if ($this->request->getParam('copy_products')) {
            $this->copyProducts($toCategory, $fromCategory->getProductsPosition());
        }

        //fix for unnecessary duplicating one child if ($fromCategoryId == $toParentId) part #2
        if ($fromCategoryId == $toParentId) {
            $this->notForCopy[] = $toCategory->getId();
        }

        $this->copyStoreData((int)$fromCategory->getId(), (int)$toCategory->getId());

        return (int)$toCategory->getId();
    }

    private function save(Category $toCategory, $isProductsSave = false): void
    {
        try {
            $toCategory->save();
        } catch (\Magento\UrlRewrite\Model\Exception\UrlAlreadyExistsException $e) {
            if (!$isProductsSave) {
                $newUrlKey = $this->suggestUrlKey($toCategory->getUrlKey(), $toCategory->getId());
                $toCategory->setUrlKey($newUrlKey);
                $this->save($toCategory);
            } else {
                $urls = $e->getUrls();
                $productIds = array_unique(array_column($urls, 'entity_id'));

                $this->messageManager->addNoticeMessage(
                    __(
                        'Products with the specified IDs (%1) can\'t be added to the category '.
                        'due to conflicts with the URL rewrites IDs (%2).',
                        implode(', ', $productIds),
                        implode(', ', array_keys($urls))
                    )
                );

                $this->copyProducts($toCategory, $toCategory->getPostedProducts() ?? [], array_flip($productIds));
            }
        }
    }

    private function suggestUrlKey(string $urlKey, $catId = null): string
    {
        if ($catId) {
            $urlKey .= '-' . $catId;
        } elseif (preg_match('/(.+)-(\d+)$/', $urlKey)) {
            $urlKey = ++$urlKey;
        } else {
            $urlKey .= '-1';
        }

        return $urlKey;
    }

    private function copyProducts(Category $toCategory, $postedProducts = [], $affectedProductIds = []): void
    {
        if ($affectedProductIds) {
            $postedProducts = array_diff_key($postedProducts, $affectedProductIds);
        }

        $toCategory->setPostedProducts($postedProducts);
        $this->save($toCategory, true);
    }

    private function copyStoreData(int $fromCategoryId, int $toCategoryId): void
    {
        $stores = $this->storeManager->getStores();

        if (!empty($stores)) {
            foreach ($stores as $store) {
                $this->copyCategoryData($fromCategoryId, $toCategoryId, (int)$store->getId());
            }
        }
    }

    private function copyCategoryData(int $fromCategoryId, int $toCategoryId, int $storeId): void
    {
        /** @var Category $fromCategory */
        $fromCategory = $this->categoryRepository->get($fromCategoryId, $storeId);
        $fromCategoryData = $fromCategory->getData();

        /** @var Category $toCategory */
        $toCategory = $this->categoryRepository->get($toCategoryId, $storeId);

        foreach ($this->unsetArray as $field) {
            unset($fromCategoryData[$field]);
        }

        $toCategory->addData($fromCategoryData);

        $this->searchAndReplace($toCategory);

        $this->save($toCategory);

        if ($this->metadata->getEdition() == 'Enterprise' &&
            $this->request->getParam('copy_rules')
        ) {
            $this->saveCategoryRules($fromCategory, $toCategory, $storeId);
        }
    }

    private function searchAndReplace(Category $category): void
    {
        $fieldsToReplaceIn = [
            'name',
            'description',
            'meta_title',
            'meta_keywords',
            'meta_description',
        ];

        $search = $this->request->getParam('search', null);
        $replace = $this->request->getParam('replace', null);

        if (!$search || !$replace) {
            return;
        }

        foreach ($fieldsToReplaceIn as $field) {
            if (null !== $category->getData($field) &&
                $value = $category->getData($field)
            ) {
                foreach ($search as $i => $searchEntity) {
                    if ($searchEntity && isset($replace[$i])) {
                        $value = str_replace($searchEntity, $replace[$i], $value);
                    }
                }
                $category->setData($field, $value);
            }
        }
    }

    private function saveCategoryRules(Category $fromCategory, Category $toCategory, int $storeId): void
    {
        if ($this->request->getParam('copy_rules_child') || !$this->savedParentCategoryRule) {
            /** @var \Magento\VisualMerchandiser\Model\Rules $visualMerchandiser */
            $visualMerchandiser = $this->objectManager->create(\Magento\VisualMerchandiser\Model\Rules::class);
            /** @var \Magento\VisualMerchandiser\Model\Rules $rules */
            $rules = $visualMerchandiser->loadByCategory($fromCategory);

            if ($rules->getData()) {
                if ($rules->getStoreId() == $storeId) {
                    $rules->setId(null);
                    $rules->setCategoryId($toCategory->getId());
                    $rules->save();
                }
            }

            $this->savedParentCategoryRule = true;
        }
    }
}
