<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Amasty Related Products (Most Viewed) GraphQl for Magento 2 (System)
 */

namespace Amasty\MostviewedGraphQl\Test\GraphQl;

use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Framework\GraphQl\Query\Uid;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

class GetAmMostviewedBundlePacksTest extends GraphQlAbstract
{
    private const MAIN_RESPONSE_KEY = 'amMostviewedBundlePacks';

    /**
     * @var ProductRepositoryInterface
     */
    private $productRepository;

    /**
     * @var Uid
     */
    private $idEncoder;

    protected function setUp(): void
    {
        parent::setUp();

        $objectManager = Bootstrap::getObjectManager();
        $this->productRepository = $objectManager->get(ProductRepositoryInterface::class);
        $this->idEncoder = $objectManager->create(Uid::class);
    }

    /**
     * @group amasty_mostviewed
     *
     * @magentoConfigFixture base_website ammostviewed/bundle_packs/display_cart_block 1
     * @magentoConfigFixture base_website ammostviewed/bundle_packs/display_cart_message 0
     * @magentoConfigFixture base_website ammostviewed/bundle_packs/top_menu_enabled 1
     *
     * @magentoApiDataFixture Amasty_MostviewedGraphQl::Test/GraphQl/_files/create_mostviewed_pack.php
     */
    public function testAmMostviewedBundlePacks(): void
    {
        $simpleId = $this->productRepository->get('simple_product')->getId();
        $simpleUid = $this->idEncoder->encode((string)$simpleId);

        $assertArray = [
            'is_display_cart_block' => true,
            'is_display_cart_message' => false,
            'is_top_menu_enabled' => 1,
            'items' => [
                [
                    'apply_for_parent' => true,
                    'block_title' => 'Amasty Test Buy together',
                    'cart_message' => 'Amasty Test Cart Message',
                    'discount_amount' => 10,
                    'discount_type' => 1,
                    'items' => [
                        [
                            'product' => [
                                'name' => 'Simple Product',
                            ],
                            'qty' => 1,
                            'discount_amount' => null
                        ],
                        [
                            'product' => [
                                'name' => 'Virtual Product',
                            ],
                            'qty' => 1,
                            'discount_amount' => null
                        ]
                    ]
                ]
            ],
            'main_product' => [
                'name' => 'Simple Product'
            ]
        ];

        $response = $this->graphQlQuery($this->getQuery($simpleUid));

        $this->assertArrayHasKey(self::MAIN_RESPONSE_KEY, $response);
        $this->assertResponseFields($response[self::MAIN_RESPONSE_KEY], $assertArray);
    }

    private function getQuery(string $productUid): string
    {
        return <<<QUERY
query {
  amMostviewedBundlePacks(uid: "$productUid") {
    is_display_cart_block
    is_display_cart_message
    is_top_menu_enabled
    items {
      apply_for_parent
      block_title
      cart_message
      discount_amount
      discount_type
      items {
        discount_amount
        product {
          name
        }
        qty
      }
    }
    main_product {
      name
    }
  }
}
QUERY;
    }
}
