<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Amasty Related Products (Most Viewed) GraphQl for Magento 2 (System)
 */

namespace Amasty\MostviewedGraphQl\Model\Resolver\BundlePacks;

use Amasty\Mostviewed\Api\Data\PackInterface;
use Amasty\MostviewedGraphQl\Model\ProductInfo;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory;
use Magento\CatalogGraphQl\Model\Resolver\Products\DataProvider\Product\CollectionPostProcessor;
use Magento\CatalogGraphQl\Model\Resolver\Products\DataProvider\Product\CollectionProcessorInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\GraphQl\Query\FieldTranslator;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;

class GetProductsData
{
    /**
     * @var CollectionFactory
     */
    private $productCollectionFactory;

    /**
     * @var FieldTranslator
     */
    private $fieldTranslator;

    /**
     * @var SearchCriteriaBuilder
     */
    private $searchCriteriaBuilder;

    /**
     * @var CollectionProcessorInterface
     */
    private $collectionPreProcessor;

    /**
     * @var CollectionPostProcessor
     */
    private $collectionPostProcessor;

    /**
     * @var ProductInfo
     */
    private $productInfo;

    public function __construct(
        CollectionFactory $productCollectionFactory,
        FieldTranslator $fieldTranslator,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        CollectionProcessorInterface $collectionPreProcessor,
        CollectionPostProcessor $collectionPostProcessor,
        ProductInfo $productInfo
    ) {
        $this->productCollectionFactory = $productCollectionFactory;
        $this->fieldTranslator = $fieldTranslator;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->collectionPreProcessor = $collectionPreProcessor;
        $this->collectionPostProcessor = $collectionPostProcessor;
        $this->productInfo = $productInfo;
    }

    /**
     * @param PackInterface $pack
     * @param ResolveInfo $info
     * @param int[] $ids
     */
    public function execute(PackInterface $pack, ResolveInfo $info, array $ids = []): array
    {
        $data = [];
        $productCollection = $this->productCollectionFactory->create();
        $productCollection->addIdFilter($ids);

        $attributes = [];
        $productFields = $info->getFieldSelection(3);
        if (isset($productFields['items']['items']['product'])) {
            foreach (array_keys($productFields['items']['items']['product']) as $fieldName) {
                $attributes[] = $this->fieldTranslator->translate($fieldName);
            }
        }

        $this->searchCriteriaBuilder->addFilter('entity_id', $ids, 'in');

        $this->collectionPreProcessor->process($productCollection, $this->searchCriteriaBuilder->create(), $attributes);
        $productCollection->getSelect()->order(new \Zend_Db_Expr('FIELD(e.entity_id,' . implode(',', $ids).')'));
        $productCollection->load();
        $this->collectionPostProcessor->process($productCollection, $attributes);

        foreach ($productCollection->getItems() as $key => $product) {
            $data[$key]['product'] = $this->productInfo->getProductInfo($product);
            $data[$key]['qty'] = $pack->getChildProductQty((int) $product->getData('entity_id'));
            $data[$key]['discount_amount'] = $pack->getChildProductDiscount((int) $product->getData('entity_id'));
        }

        return $data;
    }
}
