<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Amasty Related Products (Most Viewed) GraphQl for Magento 2 (System)
 */

namespace Amasty\MostviewedGraphQl\Model\Resolver;

use Amasty\Mostviewed\Api\Data\PackInterface;
use Amasty\Mostviewed\Model\Customer\CustomerGroupContextInterface;
use Amasty\MostviewedGraphQl\Model\ProductInfo;
use Amasty\MostviewedGraphQl\Model\Resolver\BundlePacks\GetProductsData;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\GraphQl\Query\Uid;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory;

class BundlePacks implements ResolverInterface
{
    /**
     * @var \Amasty\Mostviewed\Block\Widget\Related
     */
    private $bundlePack;

    /**
     * @var \Magento\Framework\Registry
     */
    private $registry;

    /**
     * @var \Magento\Catalog\Api\ProductRepositoryInterface
     */
    private $productRepository;

    /**
     * @var ProductInfo
     */
    private $productInfo;

    /**
     * @var \Amasty\Mostviewed\Helper\Config
     */
    private $config;

    /**
     * @var \Amasty\Mostviewed\Block\Checkout\Cart\Messages
     */
    private $messages;

    /**
     * @var \Amasty\Mostviewed\Model\ConfigProvider
     */
    private $configProvider;

    /**
     * @var Uid
     */
    private $uidEncoder;

    /**
     * @var GetProductsData
     */
    private $getProductsData;

    /**
     * @var CustomerGroupContextInterface
     */
    private $customerGroupContext;

    public function __construct(
        \Amasty\Mostviewed\Block\Product\BundlePack $bundlePack,
        \Amasty\Mostviewed\Helper\Config $config,
        \Amasty\Mostviewed\Model\ConfigProvider $configProvider,
        \Magento\Framework\Registry $registry,
        \Magento\Catalog\Api\ProductRepositoryInterface $productRepository,
        ProductInfo $productInfo,
        CollectionFactory $productCollectionFactory, // @deprecated
        \Amasty\Mostviewed\Block\Checkout\Cart\Messages $messages,
        Uid $uidEncoder,
        ?GetProductsData $getProductsData = null,
        ?CustomerGroupContextInterface $customerGroupContext = null
    ) {
        $this->bundlePack = $bundlePack;
        $this->registry = $registry;
        $this->productRepository = $productRepository;
        $this->productInfo = $productInfo;
        $this->config = $config;
        $this->messages = $messages;
        $this->configProvider = $configProvider;
        $this->uidEncoder = $uidEncoder;
        $this->getProductsData = $getProductsData ?? ObjectManager::getInstance()->get(GetProductsData::class);
        $this->customerGroupContext = $customerGroupContext ?? ObjectManager::getInstance()->get(
            CustomerGroupContextInterface::class
        );
    }

    /**
     * @param Field $field
     * @param \Magento\Framework\GraphQl\Query\Resolver\ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return array|\Magento\Framework\GraphQl\Query\Resolver\Value|mixed
     * @throws \Exception
     */
    public function resolve(
        Field $field,
        $context,
        ResolveInfo $info,
        array $value = null,
        array $args = null
    ) {
        try {
            $this->customerGroupContext->set((int)$context->getExtensionAttributes()->getCustomerGroupId());
            $productId = $this->uidEncoder->decode($args['uid']);
            $product = $this->productRepository->getById($productId);
            $this->registry->register('product', $product);
        } catch (\Exception $e) {
            return ['error' => 'Wrong parameters.'];
        }

        $data = $this->getData($info);
        $data['main_product'] = $this->productInfo->getProductInfo($product);

        return $data ?? [];
    }

    private function getData(ResolveInfo $info): array
    {
        if ($this->bundlePack->isBundlePacksExists()) {
            $data = $this->getConfigData();
            $packs = $this->bundlePack->getBundles();
            foreach ($packs as $pack) {
                $data['items'][] = $this->getBundlePackData($pack, $info);
            }
        }

        return $data ?? [];
    }

    /**
     * @return array
     */
    private function getConfigData(): array
    {
        return [
            'is_top_menu_enabled' => $this->config->isTopMenuEnabled(),
            'is_display_cart_message' => $this->configProvider->isMessageInCartEnabled(),
            'is_display_cart_block' => $this->config->isBlockInCartEnabled()
        ];
    }

    private function getBundlePackData(PackInterface $pack, ResolveInfo $info): array
    {
        $data = $pack->getData();
        if ($messageObject = $this->messages->getMessage()) {
            $data['cart_message'] = $this->messages->convertMessage($messageObject);
        }
        $data['items'] = $this->getProductsData->execute(
            $pack,
            $info,
            explode(',', $pack->getData('product_ids'))
        );
        $data['model'] = $pack;

        return $data;
    }
}
