/**
 *  Amasty Popup widget
 */

define([
    'jquery',
    'underscore'
], function ($, _) {
    'use strict';

    $.widget('am.stockStatusPopup', {
        options: {
            defaultStocksData: "",
            childsData: {}
        },
        selectors: {
            openButton: '[data-amstock-js="open-popup"]',
            closeButton: '[data-amstock-js="close-popup"]',
            popupBlock: '[data-amstock-js="popup-block"]',
            popupHeader: '[data-amstock-js="popup-header"]',
            popupContent: '[data-amstock-js="popup-content"]',
            dropdowns: 'select.super-attribute-select, select.swatch-select',
            swatchOption: 'div.swatch-option'
        },
        classes: {
            active: '-active',
            popupActive: '-amstock-popup-active'
        },
        nodes: {
            triggerButtons: null,
            closeButton: null,
            popupBlock: null,
            popupHeader: null,
            popupContent: null,
            bodyElement: null
        },

        /**
         * @private
         * @returns {void}
         */
        _create: function () {
            this._initNodes();
            this._initListeners();
            this._configureSwatchOptions();
        },

        /**
         * @private
         * @returns {void}
         */
        _configureSwatchOptions: function () {
            var self = this;


            $(document).on('swatch.initialized', function() {
                self.nodes.bodyElement.on(
                    {
                        'click': function () {
                            self.onConfigure();
                        }
                    },
                    self.selectors.swatchOption + ', ' + self.selectors.dropdowns
                );
                self.onConfigure();
            });

            $(document).on('configurable.initialized', function() {
                self.nodes.bodyElement.on(
                    {
                        'change': function () {
                            self.onConfigure();
                        }
                    },
                    self.selectors.swatchOption + ', ' + self.selectors.dropdowns
                );
                self.onConfigure();
            });
        },

        /**
         * @public
         * @returns {void}
         */
        onConfigure: function () {
            var selectedKey,
                trimSelectedKey;

            selectedKey = this.getCurrentSelectedKey();
            trimSelectedKey = selectedKey.substr(0, selectedKey.length - 1);

            if ('undefined' !== typeof(this.options.childsData[trimSelectedKey])) {
                this.nodes.popupContent[0].innerHTML = this.options.childsData[trimSelectedKey];
            } else {
                this.nodes.popupContent[0].innerHTML = this.options.defaultStocksData;
            }

            this._calculateContentHeight();
        },

        /**
         * @public
         * @returns {String}
         */
        getCurrentSelectedKey: function () {
            var result = '',
                optionId;

            this.settingsForKey = $(this.selectors.dropdowns + ', ' + this.selectors.swatchOption + '.selected').not('.slick-cloned');

            if (this.settingsForKey.length) {
                for (var i = 0; i < this.settingsForKey.length; i++) {
                    if (parseInt(this.settingsForKey[i].value) > 0) {
                        result += this.settingsForKey[i].value + ',';
                    }

                    optionId = $(this.settingsForKey[i]).attr('option-id') || $(this.settingsForKey[i]).data('option-id');

                    if (parseInt(optionId) > 0) {
                        result += optionId + ',';
                    }
                }
            }

            return result;
        },

        /**
         * Calculate and set maxHeight to the popup content block
         * @private
         * @returns {void}
         */
        _calculateContentHeight: function () {
            this.nodes.popupContent.css({
                'maxHeight': this.nodes.popupBlock.height() - this.nodes.popupHeader.outerHeight(true)
            });
        },

        /**
         * @private
         * @returns {void}
         */
        _initNodes: function () {
            this.nodes.triggerButtons = $(this.selectors.openButton);
            this.nodes.closeButton = $(this.selectors.closeButton, this.element);
            this.nodes.popupBlock = $(this.selectors.popupBlock, this.element);
            this.nodes.popupHeader = $(this.selectors.popupHeader, this.element);
            this.nodes.popupContent = $(this.selectors.popupContent, this.element);
            this.nodes.bodyElement = $('body');
        },

        /**
         * @private
         * @returns {void}
         */
        _initListeners: function () {
            var self = this;

            if (!this.nodes.triggerButtons) {
                return;
            }

            this.nodes.triggerButtons.on('click', function () {
                self.onConfigure();
                self._toggle(true);
            });

            this.nodes.closeButton.on('click', function () {
                self._toggle(false);
            });

            $(window).on('resize', _.debounce(function () {
                self._calculateContentHeight();
            }, 100));
        },

        /**
         * @param {Boolean} state
         * @private
         * @returns {void}
         */
        _toggle: function (state) {
            this.element.toggleClass(this.classes.active, state);
            $('body').toggleClass(this.classes.popupActive, state);
        }
    });

    return $.am.stockStatusPopup;
});
