<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Custom Stock Status MSI for Magento 2 (System)
 */

namespace Amasty\CustomStockStatusMsi\Model\Stockstatus\Specification;

use Amasty\CustomStockStatusMsi\Model\ConfigProvider;
use Amasty\CustomStockStatusMsi\Model\Inventory\GetPrioritySource;
use Amasty\CustomStockStatusMsi\Model\Range\GetMsiRangesForRuleAndQty;
use Amasty\CustomStockStatusMsi\Model\Range\GetProductQtyAdaptForSourceRange;
use Amasty\Stockstatus\Model\Range\GetTargetStatusId;
use Amasty\Stockstatus\Model\Range\IsAvailableForProduct;
use Amasty\Stockstatus\Model\Rule\GetRuleForProduct;
use Magento\Catalog\Api\Data\ProductInterface;
use Amasty\Stockstatus\Model\Stockstatus\Specification\SpecificationInterface;

class PrioritySourceStatus implements SpecificationInterface
{
    /**
     * @var GetRuleForProduct
     */
    private $getRuleForProduct;

    /**
     * @var GetPrioritySource
     */
    private $getPrioritySource;

    /**
     * @var GetMsiRangesForRuleAndQty
     */
    private $getMsiRangesForRuleAndQty;

    /**
     * @var GetProductQtyAdaptForSourceRange
     */
    private $getProductQtyAdaptForSourceRange;

    /**
     * @var GetTargetStatusId
     */
    private $getTargetStatusId;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var IsAvailableForProduct
     */
    private $isAvailableForProduct;

    public function __construct(
        ConfigProvider $configProvider,
        GetRuleForProduct $getRuleForProduct,
        GetPrioritySource $getPrioritySource,
        GetMsiRangesForRuleAndQty $getMsiRangesForRuleAndQty,
        GetProductQtyAdaptForSourceRange $getProductQtyAdaptForSourceRange,
        GetTargetStatusId $getTargetStatusId,
        IsAvailableForProduct $isAvailableForProduct
    ) {
        $this->getRuleForProduct = $getRuleForProduct;
        $this->getPrioritySource = $getPrioritySource;
        $this->getMsiRangesForRuleAndQty = $getMsiRangesForRuleAndQty;
        $this->getProductQtyAdaptForSourceRange = $getProductQtyAdaptForSourceRange;
        $this->getTargetStatusId = $getTargetStatusId;
        $this->configProvider = $configProvider;
        $this->isAvailableForProduct = $isAvailableForProduct;
    }

    public function resolve(ProductInterface $product): ?int
    {
        if (!$this->configProvider->isReplaceDefaultWithSourceStatus()
            || !$this->isAvailableForProduct->execute($product)
        ) {
            return null;
        }

        $appliedRule = $this->getRuleForProduct->execute((int) $product->getId(), $product->getStoreId());

        if ($appliedRule && $appliedRule->isActivateMsiQtyRanges()) {
            $product->getExtensionAttributes()->getStockstatusInformation()->setRuleId((int) $appliedRule->getId());

            if ($sourceCode = $this->getPrioritySource->execute($product->getData('sku'))) {
                $ranges = $this->getMsiRangesForRuleAndQty->execute(
                    $sourceCode,
                    (int)$appliedRule->getId(),
                    $this->getProductQtyAdaptForSourceRange->execute($product, $sourceCode)
                );
                $statusId = $this->getTargetStatusId->execute($ranges);
            }
        }

        return $statusId ?? null;
    }
}
