<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Custom Stock Status MSI for Magento 2 (System)
 */

namespace Amasty\CustomStockStatusMsi\Model\Stockstatus;

use Amasty\CustomStockStatusMsi\Api\Data\SourceStatusInformationInterface;
use Amasty\CustomStockStatusMsi\Api\Data\SourceStatusInformationInterfaceFactory;
use Amasty\Stockstatus\Api\Icon\GetIconUrlInterface;
use Amasty\CustomStockStatusMsi\Model\ConfigProvider;
use Amasty\CustomStockStatusMsi\Model\Range\GetMsiRangesForRuleAndQty;
use Amasty\CustomStockStatusMsi\Model\Range\GetProductQtyAdaptForSourceRange;
use Amasty\Stockstatus\Model\Range\GetTargetStatusId;
use Amasty\Stockstatus\Model\Rule\GetRuleForProduct;
use Amasty\Stockstatus\Model\Stockstatus\Formatter;
use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Inventory\Model\Source\Command\GetSourcesAssignedToStockOrderedByPriorityCache;
use Magento\InventoryApi\Api\Data\SourceInterface;
use Magento\InventoryCatalog\Model\GetStockIdForCurrentWebsite;

class SourceProcessor
{
    /**
     * @var GetStockIdForCurrentWebsite
     */
    private $getStockIdForCurrentWebsite;

    /**
     * @var GetSourcesAssignedToStockOrderedByPriorityCache
     */
    private $getSourcesOrderedByPriorityCache;

    /**
     * @var SourceStatusInformationInterfaceFactory
     */
    private $sourceStatusInformationInterfaceFactory;

    /**
     * @var GetMsiRangesForRuleAndQty
     */
    private $getMsiRangesForRuleAndQty;

    /**
     * @var GetProductQtyAdaptForSourceRange
     */
    private $getProductQtyAdaptForSourceRange;

    /**
     * @var GetTargetStatusId
     */
    private $getTargetStatusId;

    /**
     * @var GetRuleForProduct
     */
    private $getRuleForProduct;

    /**
     * @var Formatter
     */
    private $formatter;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var GetIconUrlInterface
     */
    private $getIconUrl;

    public function __construct(
        ConfigProvider $configProvider,
        GetStockIdForCurrentWebsite $getStockIdForCurrentWebsite,
        GetSourcesAssignedToStockOrderedByPriorityCache $getSourcesOrderedByPriorityCache,
        SourceStatusInformationInterfaceFactory $sourceStatusInformationInterfaceFactory,
        GetRuleForProduct $getRuleForProduct,
        GetMsiRangesForRuleAndQty $getMsiRangesForRuleAndQty,
        GetProductQtyAdaptForSourceRange $getProductQtyAdaptForSourceRange,
        GetTargetStatusId $getTargetStatusId,
        Formatter $formatter,
        GetIconUrlInterface $getIconUrl
    ) {
        $this->getStockIdForCurrentWebsite = $getStockIdForCurrentWebsite;
        $this->getSourcesOrderedByPriorityCache = $getSourcesOrderedByPriorityCache;
        $this->sourceStatusInformationInterfaceFactory = $sourceStatusInformationInterfaceFactory;
        $this->getRuleForProduct = $getRuleForProduct;
        $this->getMsiRangesForRuleAndQty = $getMsiRangesForRuleAndQty;
        $this->getProductQtyAdaptForSourceRange = $getProductQtyAdaptForSourceRange;
        $this->getTargetStatusId = $getTargetStatusId;
        $this->formatter = $formatter;
        $this->configProvider = $configProvider;
        $this->getIconUrl = $getIconUrl;
    }

    /**
     * Add statuses for current website MSI sources in product extensions.
     *
     * @param ProductInterface $product
     * @return void
     */
    public function execute(ProductInterface $product): void
    {
        if ($product->getExtensionAttributes()->getSourcesStatusInformation()
            || !$this->configProvider->isBreakdownEnabled()
        ) {
            return;
        }

        $this->initExtensionAttributes($product);

        if ($stockId = $this->getStockIdForCurrentWebsite->execute()) {
            $appliedRule = $this->getRuleForProduct->execute((int) $product->getId(), $product->getStoreId());

            if ($appliedRule && $appliedRule->isActivateMsiQtyRanges()) {
                foreach ($this->getEnabledSourcesOrderedByStockId($stockId) as $source) {
                    $ranges = $this->getMsiRangesForRuleAndQty->execute(
                        $source->getSourceCode(),
                        (int) $appliedRule->getId(),
                        $this->getProductQtyAdaptForSourceRange->execute($product, $source->getSourceCode())
                    );
                    if ($statusId = $this->getTargetStatusId->execute($ranges)) {
                        $this->populateExtensionAttributes($product, $source, $statusId);
                    }
                }
            }
        }
    }

    private function initExtensionAttributes(ProductInterface $product): void
    {
        $product->getExtensionAttributes()->setSourcesStatusInformation([]);
    }

    private function populateExtensionAttributes(
        ProductInterface $product,
        SourceInterface $source,
        int $statusId
    ): void {
        /** @var SourceStatusInformationInterface[] $sourcesStatusInformation */
        $sourcesStatusInformation = $product->getExtensionAttributes()->getSourcesStatusInformation();

        /** @var SourceStatusInformationInterface $sourceStatusInformation */
        $sourceStatusInformation = $this->sourceStatusInformationInterfaceFactory->create();
        $sourceStatusInformation->setSourceCode($source->getSourceCode());
        $sourceStatusInformation->setSourceLabel($source->getName());
        $sourceStatusInformation->setStatusId($statusId);
        $sourceStatusInformation->setStatusMessage(
            $this->formatter->execute($product, $statusId, [
                Formatter::SOURCE_CODE => $source->getSourceCode()
            ])
        );
        $sourceStatusInformation->setStatusIcon(
            $this->getIconUrl->execute($statusId, $product->getStoreId())
        );

        $sourcesStatusInformation[] = $sourceStatusInformation;
        $product->getExtensionAttributes()->setSourcesStatusInformation($sourcesStatusInformation);
    }

    /**
     * @param int $stockId
     * @return SourceInterface[]
     */
    private function getEnabledSourcesOrderedByStockId(int $stockId): array
    {
        $sources = $this->getSourcesOrderedByPriorityCache->execute($stockId);
        $sources = array_filter($sources, function (SourceInterface $source) {
            return $source->isEnabled();
        });
        return $sources;
    }
}
