<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Custom Stock Status MSI for Magento 2 (System)
 */

namespace Amasty\CustomStockStatusMsi\Model\Repository;

use Amasty\CustomStockStatusMsi\Api\Data\MsiRangeInterface;
use Amasty\CustomStockStatusMsi\Api\MsiRangeRepositoryInterface;
use Amasty\CustomStockStatusMsi\Model\MsiRange;
use Amasty\CustomStockStatusMsi\Model\MsiRangeFactory;
use Amasty\CustomStockStatusMsi\Model\ResourceModel\MsiRange as MsiRangeResource;
use Amasty\CustomStockStatusMsi\Model\ResourceModel\MsiRange\CollectionFactory;
use Amasty\CustomStockStatusMsi\Model\ResourceModel\MsiRange\Collection;
use Exception;
use Magento\Framework\Api\SearchResultsInterface;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Api\Search\FilterGroup;
use Magento\Ui\Api\Data\BookmarkSearchResultsInterfaceFactory;
use Magento\Framework\Api\SortOrder;

/**
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects)
 */
class MsiRangeRepository implements MsiRangeRepositoryInterface
{
    /**
     * @var BookmarkSearchResultsInterfaceFactory
     */
    private $searchResultsFactory;

    /**
     * @var MsiRangeFactory
     */
    private $msiRangeFactory;

    /**
     * @var MsiRangeResource
     */
    private $msiRangeResource;

    /**
     * Model data storage
     *
     * @var array
     */
    private $msiRanges;

    /**
     * @var CollectionFactory
     */
    private $msiRangeCollectionFactory;

    public function __construct(
        BookmarkSearchResultsInterfaceFactory $searchResultsFactory,
        MsiRangeFactory $msiRangeFactory,
        MsiRangeResource $msiRangeResource,
        CollectionFactory $msiRangeCollectionFactory
    ) {
        $this->searchResultsFactory = $searchResultsFactory;
        $this->msiRangeFactory = $msiRangeFactory;
        $this->msiRangeResource = $msiRangeResource;
        $this->msiRangeCollectionFactory = $msiRangeCollectionFactory;
    }

    /**
     * @inheritdoc
     */
    public function save(MsiRangeInterface $msiRange): MsiRangeInterface
    {
        try {
            if ($msiRange->getId()) {
                $msiRange = $this->getById($msiRange->getId())->addData($msiRange->getData());
            }
            $this->msiRangeResource->save($msiRange);
            unset($this->msiRanges[$msiRange->getId()]);
        } catch (Exception $e) {
            if ($msiRange->getId()) {
                throw new CouldNotSaveException(
                    __(
                        'Unable to save msiRange with ID %1. Error: %2',
                        [$msiRange->getId(), $e->getMessage()]
                    )
                );
            }
            throw new CouldNotSaveException(__('Unable to save new msiRange. Error: %1', $e->getMessage()));
        }

        return $msiRange;
    }

    /**
     * @inheritdoc
     */
    public function getNew(): MsiRangeInterface
    {
        return $this->msiRangeFactory->create();
    }

    /**
     * @inheritdoc
     */
    public function getById($id): MsiRangeInterface
    {
        if (!isset($this->msiRanges[$id])) {
            /** @var MsiRange $msiRange */
            $msiRange = $this->msiRangeFactory->create();
            $this->msiRangeResource->load($msiRange, $id);
            if (!$msiRange->getId()) {
                throw new NoSuchEntityException(__('MsiRange with specified ID "%1" not found.', $id));
            }
            $this->msiRanges[$id] = $msiRange;
        }

        return $this->msiRanges[$id];
    }

    /**
     * @inheritdoc
     */
    public function delete(MsiRangeInterface $msiRange): bool
    {
        try {
            $this->msiRangeResource->delete($msiRange);
            unset($this->msiRanges[$msiRange->getId()]);
        } catch (Exception $e) {
            if ($msiRange->getId()) {
                throw new CouldNotDeleteException(
                    __(
                        'Unable to remove msiRange with ID %1. Error: %2',
                        [$msiRange->getId(), $e->getMessage()]
                    )
                );
            }
            throw new CouldNotDeleteException(__('Unable to remove msiRange. Error: %1', $e->getMessage()));
        }

        return true;
    }

    /**
     * @inheritdoc
     */
    public function deleteById($id): bool
    {
        $msiRangeModel = $this->getById($id);
        $this->delete($msiRangeModel);

        return true;
    }

    /**
     * @param int $ruleId
     * @return bool
     * @throws CouldNotDeleteException
     */
    public function removeByRuleId(int $ruleId): bool
    {
        try {
            $this->msiRangeResource->deleteByRuleId($ruleId);
        } catch (\Exception $e) {
            throw new CouldNotDeleteException(
                __(
                    'Unable to remove ranges for Rule with ID %1. Error: %2',
                    [$ruleId, $e->getMessage()]
                )
            );
        }

        return true;
    }

    /**
     * @inheritdoc
     */
    public function getList(SearchCriteriaInterface $searchCriteria): SearchResultsInterface
    {
        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($searchCriteria);

        /** @var \Amasty\CustomStockStatusMsi\Model\ResourceModel\MsiRange\Collection $msiRangeCollection */
        $msiRangeCollection = $this->msiRangeCollectionFactory->create();

        // Add filters from root filter group to the collection
        foreach ($searchCriteria->getFilterGroups() as $group) {
            $this->addFilterGroupToCollection($group, $msiRangeCollection);
        }

        $searchResults->setTotalCount($msiRangeCollection->getSize());
        $sortOrders = $searchCriteria->getSortOrders();

        if ($sortOrders) {
            $this->addOrderToCollection($sortOrders, $msiRangeCollection);
        }

        $msiRangeCollection->setCurPage($searchCriteria->getCurrentPage());
        $msiRangeCollection->setPageSize($searchCriteria->getPageSize());

        $msiRanges = [];
        /** @var MsiRangeInterface $msiRange */
        foreach ($msiRangeCollection->getItems() as $msiRange) {
            $msiRanges[] = $this->getById($msiRange->getId());
        }

        $searchResults->setItems($msiRanges);

        return $searchResults;
    }

    /**
     * Helper function that adds a FilterGroup to the collection.
     *
     * @param FilterGroup $filterGroup
     * @param Collection  $msiRangeCollection
     *
     * @return void
     */
    private function addFilterGroupToCollection(FilterGroup $filterGroup, Collection $msiRangeCollection): void
    {
        foreach ($filterGroup->getFilters() as $filter) {
            $condition = $filter->getConditionType() ?: 'eq';
            $msiRangeCollection->addFieldToFilter($filter->getField(), [$condition => $filter->getValue()]);
        }
    }

    /**
     * Helper function that adds a SortOrder to the collection.
     *
     * @param SortOrder[] $sortOrders
     * @param Collection  $msiRangeCollection
     *
     * @return void
     */
    private function addOrderToCollection($sortOrders, Collection $msiRangeCollection): void
    {
        /** @var SortOrder $sortOrder */
        foreach ($sortOrders as $sortOrder) {
            $field = $sortOrder->getField();
            $msiRangeCollection->addOrder(
                $field,
                ($sortOrder->getDirection() == SortOrder::SORT_DESC) ? SortOrder::SORT_DESC : SortOrder::SORT_ASC
            );
        }
    }
}
