<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Custom Stock Status MSI for Magento 2 (System)
 */

namespace Amasty\CustomStockStatusMsi\Model\Range;

use Amasty\CustomStockStatusMsi\Api\Data\MsiRangeInterface;
use Amasty\CustomStockStatusMsi\Api\MsiRangeRepositoryInterface;
use Amasty\CustomStockStatusMsi\Model\Source\SourceCodes;
use Magento\Framework\Api\SearchCriteria;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Api\SortOrderBuilder;

class GetMsiRangesForRuleAndQty
{
    /**
     * @var MsiRangeRepositoryInterface
     */
    private $msiRangeRepository;

    /**
     * @var SearchCriteriaBuilder
     */
    private $searchCriteriaBuilder;

    /**
     * @var SortOrderBuilder
     */
    private $sortOrderBuilder;

    public function __construct(
        MsiRangeRepositoryInterface $msiRangeRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        SortOrderBuilder $sortOrderBuilder
    ) {
        $this->msiRangeRepository = $msiRangeRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->sortOrderBuilder = $sortOrderBuilder;
    }

    /**
     * @param string $sourceCode
     * @param int $ruleId
     * @param float $qty
     * @return MsiRangeInterface[]|null
     */
    public function execute(string $sourceCode, int $ruleId, float $qty): ?array
    {
        return $this->getRangesBySourceCode($sourceCode, $ruleId, $qty)
            ?: $this->getRangesBySourceCode(SourceCodes::ALL_SOURCES, $ruleId, $qty);
    }

    /**
     * @param string $sourceCode
     * @param int $ruleId
     * @param float $qty
     * @return MsiRangeInterface[]|null
     */
    private function getRangesBySourceCode(string $sourceCode, int $ruleId, float $qty): ?array
    {
        return $this->msiRangeRepository->getList($this->getSearchCriteria($sourceCode, $ruleId, $qty))->getItems();
    }

    private function getSearchCriteria(string $sourceCode, int $ruleId, float $qty): SearchCriteria
    {
        $sortByQtyFrom = $this->sortOrderBuilder
            ->setField(MsiRangeInterface::FROM)
            ->setAscendingDirection()
            ->create();
        $sortByQtyAsc = $this->sortOrderBuilder
            ->setField(MsiRangeInterface::TO)
            ->setAscendingDirection()
            ->create();
        $this->searchCriteriaBuilder->addSortOrder($sortByQtyFrom);
        $this->searchCriteriaBuilder->addSortOrder($sortByQtyAsc);

        $this->searchCriteriaBuilder->addFilter(MsiRangeInterface::SOURCE_CODE, $sourceCode);
        $this->searchCriteriaBuilder->addFilter(MsiRangeInterface::RULE_ID, $ruleId);
        $this->searchCriteriaBuilder->addFilter(MsiRangeInterface::FROM, $qty, 'lteq');
        $this->searchCriteriaBuilder->addFilter(MsiRangeInterface::TO, $qty, 'gteq');

        return $this->searchCriteriaBuilder->create();
    }
}
