<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Custom Stock Status MSI for Magento 2 (System)
 */

namespace Amasty\CustomStockStatusMsi\Model\Inventory;

use Magento\InventoryCatalog\Model\GetStockIdForCurrentWebsite;
use Magento\InventorySourceSelectionApi\Api\Data\InventoryRequestInterfaceFactory;
use Magento\InventorySourceSelectionApi\Model\SourceSelectionService;

class GetPrioritySource
{
    public const SORT_ALGORITHM = 'priority';

    /**
     * @var SourceSelectionService
     */
    private $sourceSelectionService;

    /**
     * @var InventoryRequestInterfaceFactory
     */
    private $inventoryRequestFactory;

    /**
     * @var GetStockIdForCurrentWebsite
     */
    private $getStockIdForCurrentWebsite;

    public function __construct(
        SourceSelectionService $sourceSelectionService,
        InventoryRequestInterfaceFactory $inventoryRequestFactory,
        GetStockIdForCurrentWebsite $getStockIdForCurrentWebsite
    ) {
        $this->sourceSelectionService = $sourceSelectionService;
        $this->inventoryRequestFactory = $inventoryRequestFactory;
        $this->getStockIdForCurrentWebsite = $getStockIdForCurrentWebsite;
    }

    /**
     * Return source code of priority source which have positive qty in stock,
     * else if all sources have negative qty in stock - return null
     *
     * @param string $productSku
     * @param int $qty
     * @return string|null
     */
    public function execute(string $productSku, int $qty = 1): ?string
    {
        $result = null;

        if ($stockId = $this->getStockIdForCurrentWebsite->execute()) {
            $inventoryRequest = $this->inventoryRequestFactory->create([
                'stockId' => $stockId,
                'items' => [
                    [
                        'sku' => $productSku,
                        'qty' => $qty
                    ]
                ]
            ]);

            $sourceSelectionResult = $this->sourceSelectionService->execute(
                $inventoryRequest,
                static::SORT_ALGORITHM
            );

            foreach ($sourceSelectionResult->getSourceSelectionItems() as $sourceSelectionItem) {
                if ($sourceSelectionItem->getQtyToDeduct()) {
                    $result = $sourceSelectionItem->getSourceCode();
                    break;
                }
            }
        }

        return $result;
    }
}
