<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Custom Stock Status MSI for Magento 2 (System)
 */

namespace Amasty\CustomStockStatusMsi\Model\Backend\Rule\Initialization;

use Amasty\CustomStockStatusMsi\Api\MsiRangeRepositoryInterface;
use Amasty\Stockstatus\Api\Data\RangeInterface;
use Amasty\Stockstatus\Api\Data\RuleInterface;
use Amasty\Stockstatus\Model\Backend\Rule\Initialization\RetrieveData;
use Magento\Framework\Exception\NoSuchEntityException;

class RuleMsiProcessor
{
    /**
     * @var RetrieveData
     */
    private $retrieveData;

    /**
     * @var MsiRangeRepositoryInterface
     */
    private $msiRangeRepository;

    public function __construct(
        MsiRangeRepositoryInterface $msiRangeRepository,
        RetrieveData $retrieveData
    ) {
        $this->retrieveData = $retrieveData;
        $this->msiRangeRepository = $msiRangeRepository;
    }

    public function execute(RuleInterface $rule, array $inputRuleData)
    {
        if ($rule->isActivateMsiQtyRanges()) {
            $this->populateMsiRanges($rule, $inputRuleData);
        }
    }

    private function populateMsiRanges(RuleInterface $rule, array $inputRuleData): void
    {
        /** @var RangeInterface[] $ranges */
        $ranges = [];
        $newRanges = [];
        foreach ($this->retrieveData->execute($inputRuleData, 'msi_configuration') as $rangeData) {
            $rangeId = (int)($rangeData[RangeInterface::ID] ?? 0);
            try {
                $range = $this->msiRangeRepository->getById($rangeId);
            } catch (NoSuchEntityException $e) {
                $range = $this->msiRangeRepository->getNew();
            }
            $range->addData($rangeData);
            $ranges[] = $range;
            $newRanges[] = $range->getId();
        }

        $oldRanges = $rule->getExtensionAttributes()->getMsiRanges() ?: [];
        // update ranges array with deleted ranges
        foreach ($oldRanges as $range) {
            if (!in_array($range->getId(), $newRanges)) {
                $range->isDeleted(true);
                $ranges[] = $range;
            }
        }

        $extensionAttributes = $rule->getExtensionAttributes();
        $extensionAttributes->setMsiRanges($ranges);
    }
}
