<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Custom Stock Status MSI for Magento 2 (System)
 */

namespace Amasty\CustomStockStatusMsi\Block;

use Amasty\CustomStockStatusMsi\Model\ConfigProvider;
use Amasty\CustomStockStatusMsi\Model\Stockstatus\SourceProcessor;
use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Catalog\Model\Product;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable;
use Magento\ConfigurableProduct\Model\ResourceModel\Product\Type\Configurable\Attribute\Collection;
use Magento\Framework\Api\AttributeMetadata;
use Magento\Framework\Serialize\Serializer\Json as JsonSerializer;
use Magento\Framework\View\Element\Template;
use Magento\Framework\View\Element\Template\Context;

class SourcesBreakdown extends Template
{
    /**
     * @var string
     */
    protected $_template = 'Amasty_CustomStockStatusMsi::sources_breakdown.phtml';

    private const BLOCK_NAME_PATTERN = 'amasty.customStockStatusMsi.product-%s';

    private const DEFAULT_STOCK_DATA_KEY = 'defaultStocksData';

    private const CHILDS_STOCK_DATA_KEY = 'childsData';

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var ProductInterface
     */
    private $product;

    /**
     * @var JsonSerializer
     */
    private $jsonSerializer;

    /**
     * @var SourceProcessor
     */
    private $sourceProcessor;

    public function __construct(
        ConfigProvider $configProvider,
        Context $context,
        JsonSerializer $jsonSerializer,
        SourceProcessor $sourceProcessor,
        array $data = []
    ) {
        parent::__construct($context, $data);
        $this->configProvider = $configProvider;
        $this->jsonSerializer = $jsonSerializer;
        $this->sourceProcessor = $sourceProcessor;
    }

    public function setProduct(ProductInterface $product): void
    {
        $this->product = $product;
    }

    public function getProduct(): ProductInterface
    {
        return $this->product;
    }

    public function getPopupTitle(): string
    {
        return $this->configProvider->getBreakdownTitle();
    }

    public function isSeparateLink(): bool
    {
        return $this->configProvider->isBreakdownUseSeparateLink();
    }

    public function getLinkLabel(): string
    {
        return $this->configProvider->getBreakdownLabel();
    }

    public function getSourcesInformation(): array
    {
        return $this->getProduct()->getExtensionAttributes()->getSourcesStatusInformation();
    }

    public function getSourcesInformationForPopup(): string
    {
        $result = [];
        $layout = $this->getLayout();
        $product = $this->getProduct();
        $blockSourcesData = $layout->createBlock(
            SourcesData::class,
            sprintf(self::BLOCK_NAME_PATTERN, $product->getId())
        );
        $blockSourcesData->setProduct($product);
        $result[self::DEFAULT_STOCK_DATA_KEY] = $blockSourcesData->toHtml();

        if ($product->getTypeId() === Configurable::TYPE_CODE) {
            $result[self::CHILDS_STOCK_DATA_KEY] = [];
            $childProducts = $product->getTypeInstance()->getUsedProducts($product);
            $attributes = $product->getTypeInstance()->getConfigurableAttributes($product);

            /** @var ProductInterface $product */
            foreach ($childProducts as $childProduct) {
                if (!$childProduct->getExtensionAttributes()->getSourcesStatusInformation()) {
                    $this->sourceProcessor->execute($childProduct);
                }

                $key = $this->getKey($attributes, $childProduct);

                if ($key) {
                    $blockSourcesData = $layout->createBlock(
                        SourcesData::class,
                        sprintf(self::BLOCK_NAME_PATTERN, $childProduct->getId())
                    );
                    $blockSourcesData->setProduct($childProduct);
                    $result[self::CHILDS_STOCK_DATA_KEY][$key] = $blockSourcesData->toHtml();
                }
            }
        }

        return $this->jsonSerializer->serialize($result);
    }

    private function getKey(Collection $attributes, Product $product): string
    {
        $key = [];

        foreach ($attributes as $attribute) {
            $key[] = $product->getData(
                $attribute->getData('product_attribute')->getData(AttributeMetadata::ATTRIBUTE_CODE)
            );
        }

        $key = implode(',', $key);

        return $key;
    }
}
