<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
 * @package Cron Scheduler for Magento 2
 */

namespace Amasty\CronScheduler\Controller\Adminhtml\Jobs;

use Amasty\CronScheduler\Controller\Adminhtml\AbstractJobs;
use Amasty\CronScheduler\Model\Jobs;
use Amasty\CronScheduler\Model\Repository\JobsRepository;
use Magento\Cron\Model\ScheduleFactory;
use Magento\Framework\App\Response\RedirectInterface;
use Magento\Framework\Exception\CronException;
use Magento\Framework\Stdlib\DateTime\DateTime;
use Magento\Framework\Profiler\Driver\Standard\Stat;
use Psr\Log\LoggerInterface;
use Magento\Backend\App\Action;
use Magento\Cron\Model\Schedule;

class Run extends AbstractJobs
{
    /**
     * @var JobsRepository
     */
    private $jobsRepository;

    /**
     * @var DateTime
     */
    private $dateTime;

    /**
     * @var ScheduleFactory
     */
    private $scheduleFactory;

    /**
     * @var Stat
     */
    private $statProfiler;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var RedirectInterface
     */
    private $redirect;

    public function __construct(
        Action\Context $context,
        JobsRepository $jobsRepository,
        ScheduleFactory $scheduleFactory,
        DateTime $dateTime,
        Stat $statProfiler,
        LoggerInterface $logger,
        RedirectInterface $redirect
    ) {
        parent::__construct($context);
        $this->jobsRepository = $jobsRepository;
        $this->dateTime = $dateTime;
        $this->scheduleFactory = $scheduleFactory;
        $this->statProfiler = $statProfiler;
        $this->logger = $logger;
        $this->redirect = $redirect;
    }

    public function execute()
    {
        $jobId = $this->getRequest()->getParam('id');
        $job = $this->jobsRepository->getById($jobId);

        if ($job->getStatus() == Jobs::STATUS_DISABLED) {
            $this->messageManager->addErrorMessage(__('Disabled job can\'t be runed'));
            $resultRedirect = $this->resultRedirectFactory->create();
            $resultRedirect->setUrl($this->redirect->getRefererUrl());

            return $resultRedirect;
        }
        $jobCode = $job->getJobCode();
        $schedule = $this->createScheduleItem($job);
        $jobConfig = [
            'instance' => $job->getInstance(),
            'method' => $job->getMethod(),
            'name' => $jobCode,
            'schedule' => !empty($job->getModifiedSchedule()) ? $job->getModifiedSchedule() : $job->getSchedule()
        ];
        $model = $this->_objectManager->create($jobConfig['instance']);
        $callback = [$model, $jobConfig['method']];

        if (!is_callable($callback)) {
            $schedule->setStatus(Schedule::STATUS_ERROR);
            throw new CronException(
                __('Invalid callback: %1::%2 can\'t be called', $jobConfig['instance'], $jobConfig['method'])
            );
        }
        $this->startProfiling();

        try {
            $this->logger->info(sprintf('Cron Job %s is run', $jobCode));
            // phpcs:ignore Magento2.Functions.DiscouragedFunction.Discouraged
            call_user_func_array($callback, [$schedule]);
        } catch (\Throwable $e) {
            $schedule->setStatus(Schedule::STATUS_ERROR);
            $this->logger->error(sprintf(
                'Cron Job %s has an error: %s. Statistics: %s',
                $jobCode,
                $e->getMessage(),
                $this->getProfilingStat()
            ));

            if (!$e instanceof \Exception) {
                $e = new \RuntimeException(
                    'Error when running a cron job',
                    0,
                    $e
                );
            }
            throw $e;
        } finally {
            $this->stopProfiling();
        }
        $schedule->setStatus(Schedule::STATUS_SUCCESS)->setFinishedAt(date('Y-m-d H:i:s'))->save();

        $this->logger->info(sprintf(
            'Cron Job %s is successfully finished. Statistics: %s',
            $jobCode,
            $this->getProfilingStat()
        ));
        $this->messageManager->addSuccessMessage(__('The job has been run'));
        $resultRedirect = $this->resultRedirectFactory->create();
        $resultRedirect->setUrl($this->redirect->getRefererUrl());

        return $resultRedirect;
    }

    /**
     * @param \Amasty\CronScheduler\Model\Jobs $job
     *
     * @return \Magento\Cron\Model\Schedule
     */
    private function createScheduleItem($job)
    {
        $currentDateTime = $this->dateTime->date();
        $schedule = $this->scheduleFactory->create();
        $schedule->setMessages(null);
        $schedule->setJobCode($job->getJobCode());
        $schedule->setStatus(Schedule::STATUS_RUNNING);
        $schedule->setCreatedAt($currentDateTime);
        $schedule->setScheduledAt($currentDateTime);
        $schedule->setExecutedAt($currentDateTime);
        $schedule->save();

        return $schedule;
    }

    /**
     * Starts profiling
     *
     * @return void
     */
    private function startProfiling()
    {
        $this->statProfiler->clear();
        $this->statProfiler->start('job', microtime(true), memory_get_usage(true), memory_get_usage());
    }

    /**
     * Stops profiling
     *
     * @return void
     */
    private function stopProfiling()
    {
        $this->statProfiler->stop('job', microtime(true), memory_get_usage(true), memory_get_usage());
    }

    /**
     * Retrieves statistics in the JSON format
     *
     * @return string
     */
    private function getProfilingStat()
    {
        $stat = $this->statProfiler->get('job');
        unset($stat[Stat::START]);
        return json_encode($stat);
    }
}
