<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Multiple Coupons for Magento 2
*/

declare(strict_types=1);

namespace Amasty\Coupons\Test\Integration;

use Magento\TestFramework\Helper\Bootstrap;

class QuoteAmountCouponApplyTest extends \PHPUnit\Framework\TestCase
{
    /**
     * @var \Magento\TestFramework\ObjectManager
     */
    protected $objectManager;

    /**
     * @var \Magento\Framework\Registry
     */
    protected $registry;

    /**
     * @var \Amasty\Coupons\Api\ApplyCouponsToCartInterface
     */
    protected $applyCouponsToCart;

    /**
     * @var \Magento\Quote\Api\CartTotalRepositoryInterface
     */
    protected $totalRepository;

    protected function setUp(): void
    {
        $this->objectManager = Bootstrap::getObjectManager();
        $this->registry = $this->objectManager->get(\Magento\Framework\Registry::class);
        $this->applyCouponsToCart = $this->objectManager->get(\Amasty\Coupons\Api\ApplyCouponsToCartInterface::class);
        $this->totalRepository = $this->objectManager->get(\Magento\Quote\Api\CartTotalRepositoryInterface::class);
    }

    /**
     * @magentoDataFixture Magento/Sales/_files/quote.php
     * @magentoDataFixture Magento/Checkout/_files/discount_10percent.php
     * @magentoDataFixture Magento/Checkout/_files/discount_10percent_generalusers.php
     */
    public function testSingle()
    {
        /** @var \Magento\Quote\Model\Quote $quote */
        $quote = $this->objectManager->create(\Magento\Quote\Model\Quote::class);
        $quote->load('test01', 'reserved_order_id');
        $cartId = (int)$quote->getId();
        if (!$cartId) {
            $this->fail('Failed to load fixture. Quote is undefined.');
        }
        $salesRule = $this->objectManager->create(\Magento\SalesRule\Model\Rule::class);
        $salesRuleId = $this->registry->registry('Magento/Checkout/_file/discount_10percent');
        $salesRule->load($salesRuleId);

        $couponCodes = [$salesRule->getPrimaryCoupon()->getCode(), '2?ds5!2d', 'invalid_coupon'];

        $expectedAmount = $quote->getBaseSubtotal() - ($quote->getBaseSubtotal() * 0.1);

        $this->applyCouponsToCart->apply($cartId, $couponCodes);

        $this->assertSame(
            $salesRule->getPrimaryCoupon()->getCode(),
            $this->totalRepository->get($cartId)->getCouponCode()
        );

        $this->assertSame(
            (float)$expectedAmount,
            (float)$this->totalRepository->get($cartId)->getBaseGrandTotal(),
            'Discount throw coupon code isn\'t applied.'
        );
    }

    public static function generatedCouponsFixture()
    {
        /** @var \Magento\SalesRule\Model\Rule $salesRule */
        $salesRule = Bootstrap::getObjectManager()->create(\Magento\SalesRule\Model\Rule::class);
        $salesRuleId = Bootstrap::getObjectManager()->get(\Magento\Framework\Registry::class)
            ->registry('Magento/Checkout/_file/discount_10percent');
        $salesRule->load($salesRuleId);
        $salesRule->setUseAutoGeneration(1)->save();

        $coupon = Bootstrap::getObjectManager()->create(\Magento\SalesRule\Model\Coupon::class);
        $coupon->setRuleId($salesRuleId)->setCode('autogenerated_3_1')->setType(1)->save();
        $coupon = Bootstrap::getObjectManager()->create(\Magento\SalesRule\Model\Coupon::class);
        $coupon->setRuleId($salesRuleId)->setCode('autogenerated_3_2')->setType(1)->save();
    }

    /**
     * Apply multiple coupons of one rule
     *
     * @magentoDbIsolation enabled
     * @magentoAppIsolation enabled
     *
     * @magentoDataFixture Magento/Sales/_files/quote.php
     * @magentoDataFixture Magento/Checkout/_files/discount_10percent.php
     * @magentoDataFixture generatedCouponsFixture
     *
     * @magentoConfigFixture current_store amcoupons/general/allow_same_rule 1
     */
    public function testMultiCoupons()
    {
        /** @var \Magento\Quote\Model\Quote $quote */
        $quote = $this->objectManager->create(\Magento\Quote\Model\Quote::class);
        $quote->load('test01', 'reserved_order_id');
        $cartId = (int)$quote->getId();
        if (!$cartId) {
            $this->fail('Failed to load fixture. Quote is undefined.');
        }
        $salesRule = $this->objectManager->create(\Magento\SalesRule\Model\Rule::class);
        $salesRuleId = $this->registry->registry('Magento/Checkout/_file/discount_10percent');
        $salesRule->load($salesRuleId);

        $couponCodes = ['autogenerated_3_1', 'autogenerated_3_2'];

        $result = $this->applyCouponsToCart->apply($cartId, $couponCodes);

        $this->assertArrayContainsSameObjectWithValue($result, 'autogenerated_3_1', true);
        $this->assertArrayContainsSameObjectWithValue($result, 'autogenerated_3_2', true);
    }

    /**
     * Apply multiple coupons of one rule
     *
     * @magentoDbIsolation enabled
     * @magentoAppIsolation enabled
     *
     * @magentoDataFixture Magento/Sales/_files/quote.php
     * @magentoDataFixture Magento/Checkout/_files/discount_10percent.php
     * @magentoDataFixture generatedCouponsFixture
     *
     * @magentoConfigFixture current_store amcoupons/general/allow_same_rule 0
     */
    public function testNoMultiCoupons()
    {
        /** @var \Magento\Quote\Model\Quote $quote */
        $quote = $this->objectManager->create(\Magento\Quote\Model\Quote::class);
        $quote->load('test01', 'reserved_order_id');
        $cartId = (int)$quote->getId();
        if (!$cartId) {
            $this->fail('Failed to load fixture. Quote is undefined.');
        }
        $salesRule = $this->objectManager->create(\Magento\SalesRule\Model\Rule::class);
        $salesRuleId = $this->registry->registry('Magento/Checkout/_file/discount_10percent');
        $salesRule->load($salesRuleId);

        $couponCodes = ['autogenerated_3_1', 'autogenerated_3_2', 'autogenerated_3_1'];

        $result = $this->applyCouponsToCart->apply($cartId, $couponCodes);

        $this->assertArrayContainsSameObjectWithValue($result, 'autogenerated_3_1', true);
        $this->assertArrayContainsSameObjectWithValue($result, 'autogenerated_3_2', false);
    }

    /**
     * @param \Amasty\Coupons\Api\Data\CouponApplyResultInterface[] $couponsResult
     * @param string $couponCode
     * @param bool $isShouldBeApplied
     */
    private function assertArrayContainsSameObjectWithValue($couponsResult, $couponCode, $isShouldBeApplied)
    {
        foreach ($couponsResult as $couponItem) {
            if ($couponItem->getCode() === $couponCode) {
                if ($couponItem->isApplied() !== $isShouldBeApplied) {
                    $this->fail(
                        'Coupon ' . $couponCode . " founded in result, but it is in wrong apply statement. " .
                        '"isApplied" Should be ' . $isShouldBeApplied ? '"true"' : '"false"' .
                        "Coupon Result Object: \n" . print_r($couponItem, true)
                    );
                }
                return;
            }
        }

        $this->fail(
            'Coupon ' . $couponCode ." is not found in result. Current Result: \n" . print_r($couponsResult, true)
        );
    }
}
