<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Color Swatches Pro for Magento 2
 */

namespace Amasty\Conf\Plugin\Checkout\Model;

use Amasty\Conf\Helper\Data;
use Magento\Catalog\Model\Product;
use Magento\Catalog\Model\ProductFactory;
use Magento\Checkout\Model\Cart as MagentoCart;
use Magento\Framework\DataObject;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Filter\LocalizedToNormalizedFactory;
use Magento\Framework\Locale\ResolverInterface;
use Magento\Store\Model\StoreManagerInterface;

class Cart
{
    public const AMCONFIGURABLE_OPTION = 'amconfigurable-option';
    public const CONFIGURABLE_OPTION = 'configurable-option';

    /**
     * @var ResolverInterface
     */
    private $locale;

    /**
     * @var ProductFactory
     */
    private $productFactory;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var Data
     */
    private $helper;

    /**
     * @var LocalizedToNormalizedFactory
     */
    private $localizedToNormalizedFactory;

    public function __construct(
        ResolverInterface $locale,
        StoreManagerInterface $storeManager,
        ProductFactory $productFactory,
        Data $helper,
        LocalizedToNormalizedFactory $localizedToNormalizedFactory
    ) {
        $this->locale = $locale;
        $this->productFactory = $productFactory;
        $this->storeManager = $storeManager;
        $this->helper = $helper;
        $this->localizedToNormalizedFactory = $localizedToNormalizedFactory;
    }

    /**
     * @param MagentoCart $subject
     * @param \Closure $closure
     * @param int|Product $productInfo
     * @param DataObject|int|array $requestInfo
     * @return MagentoCart
     */
    public function aroundAddProduct(MagentoCart $subject, \Closure $closure, $productInfo, $requestInfo)
    {
        if (isset($requestInfo[self::AMCONFIGURABLE_OPTION])) {
            $storeId = $this->storeManager->getStore()->getId();
            $flagAddedToCart = false;
            $tmpRequest = $this->getTmpRequest($requestInfo);
            foreach ($requestInfo[self::AMCONFIGURABLE_OPTION] as $optionValues) {
                try {
                    $this->prepareRequest($optionValues, $tmpRequest);
                    if ($tmpRequest['qty'] <= 0) {
                        continue;
                    }

                    if ($productInfo instanceof Product) {
                        $productInfo = $productInfo->getId();
                    }

                    //should reinitialize product( without repository! )
                    $product = $this->productFactory->create();
                    $product->setData('store_id', $storeId)->load($productInfo);

                    $closure($product, $tmpRequest);
                    $flagAddedToCart = true;
                } catch (\Exception $ex) {
                    //skip this product and add another
                    continue;
                }
            }

            $this->throwExceptionIfNeed($flagAddedToCart);
        } else {
            $closure($productInfo, $requestInfo);
        }

        return $subject;
    }

    /**
     * @param MagentoCart $subject
     * @param \Closure $closure
     * @param int $itemId
     * @param null $requestInfo
     * @param null $updatingParams
     * @return mixed
     * @throws LocalizedException
     */
    public function aroundUpdateItem(
        MagentoCart $subject,
        \Closure $closure,
        $itemId,
        $requestInfo = null,
        $updatingParams = null
    ) {
        if (isset($requestInfo[self::AMCONFIGURABLE_OPTION])) {
            $tmpRequest = $this->getTmpRequest($requestInfo);
            $flagUpdatedCart = false;
            foreach ($requestInfo[self::AMCONFIGURABLE_OPTION] as $optionValues) {
                try {
                    $this->prepareRequest($optionValues, $tmpRequest);
                    if ($tmpRequest['qty'] <= 0) {
                        continue;
                    }

                    $result = $closure($itemId, $tmpRequest, $updatingParams);
                    $flagUpdatedCart = true;
                } catch (\Exception $ex) {
                    //skip this product and add another
                    continue;
                }
            }
            $this->throwExceptionIfNeed($flagUpdatedCart);
        } else {
            $result = $closure($itemId, $requestInfo, $updatingParams);
        }

        return $result;
    }

    /**
     * @param array $requestInfo
     * @return array
     */
    private function getTmpRequest($requestInfo)
    {
        $tmpRequest = is_object($requestInfo) ? $requestInfo->getData() : $requestInfo;
        unset($tmpRequest[self::CONFIGURABLE_OPTION]);
        unset($tmpRequest[self::AMCONFIGURABLE_OPTION]);

        return $tmpRequest;
    }

    private function prepareRequest(string $optionValues, array &$tmpRequest): void
    {
        $options = $this->helper->decode($optionValues);
        $qty = $options['qty'];
        unset($options['qty']);
        $tmpRequest['super_attribute'] = [];

        foreach ($options as $attribute => $value) {
            $tmpRequest['super_attribute'][$attribute] = $value;
        }
        $localizedToNormalized = $this->localizedToNormalizedFactory->create()
            ->setOptions(['locale' => $this->locale->getLocale()]);
        $qty = $localizedToNormalized->filter($qty);
        $tmpRequest['qty'] = $qty;
    }

    /**
     * @param bool $flagChangeCart
     * @return void
     * @throws LocalizedException
     */
    private function throwExceptionIfNeed(bool $flagChangeCart): void
    {
        if (!$flagChangeCart) {
            throw new LocalizedException(__('Please specify the quantity of product(s).'));
        }
    }
}
