<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) 2023 Amasty (https://www.amasty.com)
 * @package Color Swatches Pro for Magento 2
 */

namespace Amasty\Conf\Model;

use Amasty\Base\Model\MagentoVersion;
use Amasty\Conf\Helper\Data;
use Amasty\Conf\Model\Source\MatrixMode;
use Amasty\Conf\Model\Source\Preselect;
use Magento\Catalog\Model\Product;
use Magento\CatalogInventory\Api\Data\StockItemInterface;
use Magento\CatalogInventory\Model\Stock;
use Magento\ConfigurableProduct\Block\Product\View\Type\Configurable as TypeConfigurable;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable as ConfigurableModel;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Swatches\Block\Product\Renderer\Listing\Configurable as ListingConfigurable;

class ConfigurableConfigGetter
{
    public const PLUGIN_TYPE_PRODUCT = 'product';
    public const PLUGIN_TYPE_CATEGORY = 'category';
    public const AMASTY_BACKORDER_CODE = '101';
    public const OPTION_SELECTED = 'option-selected';
    public const DATA_OPTION_SELECTED = 'data-option-selected';

    /**
     * @var Data
     */
    private $helper;

    /**
     * @var \Magento\Framework\View\LayoutFactory
     */
    private $layoutFactory;

    /**
     * @var \Magento\Catalog\Helper\Output
     */
    private $output;

    /**
     * @var \Magento\Framework\Registry
     */
    private $coreRegistry;

    /**
     * @var ConfigurableModel
     */
    private $configurableModel;

    /**
     * @var array
     */
    private $preselectedInfo = [];

    /**
     * @var \Magento\Framework\Module\Manager
     */
    private $moduleManager;

    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var \Amasty\Conf\Model\ResourceModel\Inventory
     */
    private $inventory;

    /**
     * @var MagentoVersion
     */
    private $magentoVersion;

    public function __construct(
        Data $helper,
        \Magento\Framework\View\LayoutFactory $layoutFactory,
        \Magento\Framework\Registry $registry,
        \Magento\Catalog\Helper\Output $output,
        ConfigurableModel $configurableModel,
        \Magento\Framework\Module\Manager $moduleManager,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Amasty\Conf\Model\ResourceModel\Inventory $inventory,
        MagentoVersion $magentoVersion
    ) {
        $this->helper = $helper;
        $this->layoutFactory = $layoutFactory;
        $this->output = $output;
        $this->coreRegistry = $registry;
        $this->configurableModel = $configurableModel;
        $this->moduleManager = $moduleManager;
        $this->storeManager = $storeManager;
        $this->inventory = $inventory;
        $this->magentoVersion = $magentoVersion;
    }

    /**
     * @throws NoSuchEntityException
     * @throws LocalizedException
     */
    public function execute(TypeConfigurable $subject, $isGraphQl = false): array
    {
        $config = [];

        if ($isGraphQl) {
            $config['product_page'] = $this->getConfigurableConfig($subject);
            $config['product_listing'] = $this->getListingConfigurableConfig($subject);
        } else {
            $availableNames = ['product.info.options.configurable', 'product.info.options.swatches'];
            if (in_array($subject->getNameInLayout(), $availableNames)) {
                $config = $this->getConfigurableConfig($subject);
            } elseif ($subject instanceof ListingConfigurable) {
                $config = $this->getListingConfigurableConfig($subject);
            }
        }

        return $config;
    }

    /**
     * @param TypeConfigurable $subject
     * @return array
     */
    private function getConfigurableConfig(TypeConfigurable $subject)
    {
        $config['product_information'] = $this->getProductsInformation($subject);
        $config['show_prices'] = $this->helper->getModuleConfig('general/show_price');
        $config['show_dropdown_prices'] = $this->helper->getModuleConfig('general/dropdown_price');
        $config['change_mouseover'] = $this->helper->getModuleConfig('general/change_mouseover');
        $config['selected_option_attribute_class'] = $this->getSelectedOptionAttributeClass();
        $config['show_out_of_stock'] = $this->crossOutOfStock();
        $config['swatches_slider'] = boolval($this->helper->getModuleConfig('general/swatches_slider'));
        $config['swatches_slider_items_per_view']
            = $this->helper->getModuleConfig('general/swatches_slider_items_per_view');
        $config['matrix'] = $this->isMatrixEnabled($subject->getProduct());
        $config['titles'] = $config['matrix'] ? $this->getMatrixTitles() : [];
        $preselect = $this->helper->getModuleConfig('preselect/preselect');

        if ($preselect || $subject->getProduct()->getSimplePreselect()) {
            $preselectedData = $this->getPreselectData($preselect, $subject);

            if ($preselectedData['product']) {
                $config['preselect']['product_id'] = $preselectedData['product']->getId();
                $config['preselect']['attributes'] = $preselectedData['attributes'];
            }
        }

        return $config;
    }

    /**
     * @throws NoSuchEntityException
     * @throws LocalizedException
     */
    private function getListingConfigurableConfig(TypeConfigurable $subject): array
    {
        $config['change_mouseover'] = $this->helper->getModuleConfig('general/change_mouseover');
        $config['selected_option_attribute_class'] = $this->getSelectedOptionAttributeClass();
        $config['show_out_of_stock'] = $this->crossOutOfStock();
        $config['product_information'] = $this->getProductsInformation($subject, self::PLUGIN_TYPE_CATEGORY);
        $preselect = $this->helper->getModuleConfig('preselect/preselect');
        $categoryPreselect = $this->helper->getModuleConfig('preselect/preselect_category');
        if (($preselect || $subject->getProduct()->getSimplePreselect())
            && $categoryPreselect
        ) {
            $preselectedData = $this->getPreselectData($preselect, $subject);
            if ($preselectedData['product']) {
                $config['preselect']['attributes'] = $preselectedData['attributes'];
                $config['preselect']['product_id'] = $preselectedData['product']->getId();
                $config['blockedImage'] = true;
            }
        }
        $config['preselected'] = false;

        return $config;
    }

    private function getSelectedOptionAttributeClass(): string
    {
        return version_compare($this->magentoVersion->get(), '2.4.0', '<')
            ? self::OPTION_SELECTED
            : self::DATA_OPTION_SELECTED;
    }

    /**
     * @return bool
     */
    private function crossOutOfStock()
    {
        return !$this->moduleManager->isEnabled('Amasty_Xnotif') //customers should select option
            && $this->helper->getModuleConfig('general/show_out_of_stock');
    }

    /**
     * @param Product $product
     * @return bool
     */
    private function isMatrixEnabled(\Magento\Catalog\Model\Product $product)
    {
        $setting = $this->helper->getModuleConfig('matrix/enable');

        return $setting == MatrixMode::YES_FOR_ALL
            || ($setting == MatrixMode::YES && $product->getData(Data::MATRIX_ATTRIBUTE));
    }

    /**
     * public access for Amasty_HidePrice
     *
     * @return array
     */
    private function getMatrixTitles()
    {
        $result = [
            'attribute' => __('Option'),
            'price' => __('Price'),
            'sku' => __('SKU'),
            'available' => __('Available'),
            'qty' => __('Qty'),
            'subtotal' => __('Subtotal')
        ];

        if (!$this->isShowQtyAvailable()) {
            unset($result['available']);
        }

        if (!$this->isShowSubtotal()) {
            unset($result['subtotal']);
        }

        if (!$this->isSkuDisplayed()) {
            unset($result['sku']);
        }

        return $result;
    }

    /**
     * @return bool
     */
    private function isShowQtyAvailable()
    {
        return (bool)$this->helper->getModuleConfig('matrix/available_qty');
    }

    /**
     * @return bool
     */
    private function isShowSubtotal()
    {
        return (bool)$this->helper->getModuleConfig('matrix/subtotal');
    }

    /**
     * @return bool
     */
    private function isSkuDisplayed()
    {
        return (bool)$this->helper->getModuleConfig('matrix/display_sku');
    }

    /**
     * @throws NoSuchEntityException
     * @throws LocalizedException
     */
    private function getProductsInformation(TypeConfigurable $subject, string $type = self::PLUGIN_TYPE_PRODUCT): array
    {
        $info = [];
        $reloadValues = $this->helper->getModuleConfig('reload/content');
        $reloadValues = explode(',', $reloadValues);

        $info['default'] = $this->getProductInfo($subject->getProduct(), $reloadValues, $type);

        return $this->addProductsInfo($info, $subject, $reloadValues, $type);
    }

    /**
     * @throws LocalizedException
     * @throws NoSuchEntityException
     */
    private function addProductsInfo(array $info, TypeConfigurable $subject, array $reloadValues, string $type): array
    {
        $products = $subject->getAllowProducts();
        $productsSku = $this->getProductsSku($products);
        $websiteCode = $this->storeManager->getWebsite()->getCode();
        $stockValues = $this->inventory->getStocks($productsSku, $websiteCode);
        $qtyValues = $this->inventory->getQty($productsSku, $websiteCode);

        foreach ($products as $product) {
            $productId = $product->getId();
            $productSku = $product->getSku();

            $info[$productId] = $this->getProductInfo($product, $reloadValues, $type, true);
            $info[$productId]['is_in_stock'] = (bool) ($stockValues[$productSku] ?? 0);
            $info[$productId]['qty'] = (float) ($qtyValues[$productSku] ?? 0);
        }

        return $info;
    }

    /**
     * @param array $products
     * @return array
     */
    private function getProductsSku($products = [])
    {
        $data = [];
        foreach ($products as $product) {
            $data[$product->getId()] = $product->getSku();
        }

        return $data;
    }

    /**
     * @throws LocalizedException
     * @throws NoSuchEntityException
     */
    private function getProductInfo(
        Product $product,
        array $reloadValues,
        string $type,
        bool $isNeedToCalculateMatrixSettings = false
    ): array {
        $productInfo = [];
        if ($type === self::PLUGIN_TYPE_PRODUCT && !in_array('none', $reloadValues)) {
            $layout = $this->layoutFactory->create();

            foreach ($reloadValues as $reloadValue) {
                $selector = $this->helper->getModuleConfig('reload/' . $reloadValue);
                if (!$selector) {
                    continue;
                }
                if ($reloadValue == 'attributes') {
                    $block = $layout->createBlock(
                        \Magento\Catalog\Block\Product\View\Attributes::class,
                        'product.attributes',
                        ['data' => []]
                    )->setTemplate('product/view/attributes.phtml');

                    $currentProduct = $this->coreRegistry->registry('product');
                    $this->coreRegistry->unregister('product');
                    $this->coreRegistry->register('product', $product);

                    $value = $block->setProduct($product)->toHtml();

                    $this->coreRegistry->unregister('product');
                    $this->coreRegistry->register('product', $currentProduct);
                } else {
                    $value = $this->output->productAttribute($product, $product->getData($reloadValue), $reloadValue);
                }
                if ($value) {
                    $productInfo[$reloadValue] = [
                        'selector' => $selector,
                        'value' => $value
                    ];
                }
            }
        }

        $sku = $product->getData('sku');
        $productInfo['sku_value'] = $sku;

        if ($isNeedToCalculateMatrixSettings && $this->isMatrixEnabled($product)) {
            $websiteCode = $this->storeManager->getWebsite()->getCode();
            $stockItem = $this->inventory->getStockItem($sku, $websiteCode);
            $productQty = $this->getProductQty($product->getSku(), $websiteCode);
            $productInfo['product_matrix_settings'] = [
                'need_display_out_of_stock' =>
                    $this->isNeedToDisplayOutOfStock($productQty, $stockItem, $product),
                'max_qty_limiter' => $this->getMaxQtyLimiter($productQty, $stockItem, $product)
            ];
        }

        return $productInfo;
    }

    /**
     * @throws NoSuchEntityException
     */
    private function getProductQty(string $sku, string $websiteCode): float
    {
        $productQty = $this->inventory->getQty($sku, $websiteCode);
        return is_array($productQty) ? (float) implode($productQty) : (float) $productQty;
    }

    /**
     * @throws NoSuchEntityException
     */
    private function isProductInStock(string $sku, string $websiteCode): bool
    {
        $skuValues = $this->inventory->getStocks($sku, $websiteCode);
        return is_array($skuValues) && implode($skuValues);
    }

    private function isPreorder(Product $product, StockItemInterface $stockItem): bool
    {
        return $this->helper->isPreorderEnabled($product)
            && $stockItem->getManageStock()
            && $stockItem->getBackorders() == self::AMASTY_BACKORDER_CODE;
    }

    private function isBackorder(StockItemInterface $stockItem): bool
    {
        $backorder = $stockItem->getBackorders();
        return $stockItem->getManageStock()
            && $backorder == (Stock::BACKORDERS_YES_NONOTIFY || $backorder == Stock::BACKORDERS_YES_NOTIFY);
    }

    /**
     * @throws NoSuchEntityException
     * @throws LocalizedException
     */
    private function isNeedToDisplayOutOfStock(
        float $productQty,
        StockItemInterface $stockItem,
        Product $product
    ): bool {
        $websiteCode = $this->storeManager->getWebsite()->getCode();
        $isInStock = $this->isProductInStock($product->getSku(), $websiteCode);
        $isPreorder = $this->isPreorder($product, $stockItem);
        $isBackorder = $this->isBackorder($stockItem);
        $isManageStock = $stockItem->getManageStock();

        return !$isInStock || ($isManageStock && !($isPreorder || $isBackorder) && !$productQty);
    }

    private function getMaxQtyLimiter(
        float $productQty,
        StockItemInterface $stockItem,
        Product $product
    ): float {
        $maxAllowedQty = $stockItem->getMaxSaleQty();
        $isPreorder = $this->isPreorder($product, $stockItem);
        $isBackorder = $this->isBackorder($stockItem);
        $isManageStock = $stockItem->getManageStock();

        if (($maxAllowedQty > $productQty) && !($isPreorder || $isBackorder || !$isManageStock)) {
            $maxAllowedQty = $productQty;
        }

        return $maxAllowedQty;
    }

    /**
     * @param integer $preselect
     * @param TypeConfigurable $subject
     * @return array
     */
    public function getPreselectData($preselect, $subject)
    {
        $productId = $subject->getProduct()->getId();
        if (!isset($this->preselectedInfo[$productId])) {
            $selectedProduct = $this->getSimplePreselectedChild($subject);
            if (!$selectedProduct) {
                $selectedProduct = $this->getPreselectByOption($preselect, $subject);
            }

            $this->preselectedInfo[$productId] = [
                'attributes' => $this->getAttributesForProduct($subject->getAllowAttributes(), $selectedProduct),
                'product' => $selectedProduct
            ];
        }

        return $this->preselectedInfo[$productId] ?? null;
    }

    /**
     * @param TypeConfigurable $subject
     *
     * @return Product|null
     */
    private function getSimplePreselectedChild(TypeConfigurable $subject)
    {
        $selectedProduct = null;
        if ($sku = $subject->getProduct()->getSimplePreselect()) {
            foreach ($subject->getAllowProducts() as $product) {
                if ($product->getSku() == $sku) {
                    $selectedProduct = $product;
                    break;
                }
            }
        }

        return $selectedProduct;
    }

    /**
     * @param $preselect
     * @param TypeConfigurable $subject
     *
     * @return Product|null
     */
    private function getPreselectByOption($preselect, TypeConfigurable $subject)
    {
        switch ($preselect) {
            case Preselect::FIRST_OPTIONS:
                $selectedProduct = $this->getFirstOptionProduct($subject);
                break;
            case Preselect::CHEAPEST:
                $selectedProduct = $this->getCheapestProduct($subject->getAllowProducts());
                break;
            default:
                $selectedProduct = null;
        }

        return $selectedProduct;
    }

    /**
     * @param TypeConfigurable $subject
     *
     * @return Product|null
     */
    private function getFirstOptionProduct(TypeConfigurable $subject)
    {
        $lastRow = count($subject->getAllowAttributes()) - 1;
        $selectedIdsOptions = [];

        foreach ($subject->getAllowAttributes() as $attribute) {
            $productAttribute = $attribute->getProductAttribute();
            $attributeId = $productAttribute->getId();

            if (count($selectedIdsOptions) == $lastRow) {
                foreach (($attribute['options'] ?? []) as $option) {
                    if (isset($option['value_index'])) {
                        $selectedIdsOptions[$attributeId] = $option['value_index'];

                        $selectedProduct = $this->getProductByAttributes($subject->getProduct(), $selectedIdsOptions);
                        if ($selectedProduct && $selectedProduct->hasData() && !$selectedProduct->isDisabled()) {
                            break;
                        }
                    }
                }
            } elseif (isset($attribute['options'][0]['value_index'])) {
                $selectedIdsOptions[$attributeId] = $attribute['options'][0]['value_index'];
            }
        }

        return $selectedProduct;
    }

    /**
     * Retrieve child simple product from configurable product by product attributes.
     * Use this instead of ConfigurableModel::getProductByAttributes because in ConfigurableModel
     * called productRepository and nesting level reached 256.
     *
     * @param Product $configurableProduct
     * @param array $selectedIdsOptions
     * @return Product
     */
    private function getProductByAttributes(Product $configurableProduct, array $selectedIdsOptions): Product
    {
        $productCollection = $this->configurableModel->getUsedProductCollection($configurableProduct)
            ->addAttributeToSelect('name')
            ->addAttributeToSelect('status')
            ->addAttributeToSelect('small_image');

        foreach ($selectedIdsOptions as $attributeId => $attributeValue) {
            $productCollection->addAttributeToFilter($attributeId, $attributeValue);
        }

        return $productCollection->getFirstItem();
    }

    /**
     * @param array $allowedAttributes
     * @param Product $product
     * @return array
     */
    private function getAttributesForProduct($allowedAttributes, $product)
    {
        $selectedOptions = [];
        foreach ($allowedAttributes as $attribute) {
            $attributeCode = $attribute->getProductAttribute()->getAttributeCode();
            if ($product && ($attributeValue = $product->getData($attributeCode))) {
                $selectedOptions[$attributeCode] = $attributeValue;
            }
        }

        return $selectedOptions;
    }

    /**
     * Return product with minimal price
     *
     * @param $allowedProducts
     * @return null|Product
     */
    private function getCheapestProduct($allowedProducts)
    {
        $selectedProduct = null;

        $this->coreRegistry->unregister('hideprice_off');
        $this->coreRegistry->register('hideprice_off', true);
        foreach ($allowedProducts as $product) {
            if (!$selectedProduct
                || $product->getPriceInfo()->getPrice('final_price')->getAmount()->getValue()
                < $selectedProduct->getPriceInfo()->getPrice('final_price')->getAmount()->getValue()
            ) {
                $selectedProduct = $product;
            }
        }
        $this->coreRegistry->unregister('hideprice_off');

        return $selectedProduct;
    }
}
