<?php

declare(strict_types=1);

namespace Amasty\ConfGraphQl\Model\Resolver;

use Amasty\Conf\Helper\Data;
use Amasty\Conf\Model\ConfigurableConfigGetter;
use Magento\Framework\App\Area;
use Magento\Framework\App\State;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Query\Resolver\ValueFactory;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\ConfigurableProduct\Block\Product\View\Type\Configurable as TypeConfigurable;
use Magento\Store\Model\App\Emulation;

class ConfigurableData implements ResolverInterface
{
    public const PRODUCT_PAGE_TYPE = 'product_page';
    public const PRODUCT_LISTING_TYPE = 'product_listing';

    /**
     * @var Data
     */
    private $helper;

    /**
     * @var ValueFactory
     */
    private $valueFactory;

    /**
     * @var ConfigurableConfigGetter
     */
    private $configGetter;

    /**
     * @var TypeConfigurable
     */
    private $configurable;

    /**
     * @var State
     */
    private $state;

    /**
     * @var Emulation
     */
    private $appEmulation;

    public function __construct(
        Data $helper,
        ValueFactory $valueFactory,
        ConfigurableConfigGetter $configGetter,
        TypeConfigurable $configurable,
        Emulation $appEmulation,
        State $state
    ) {
        $this->helper = $helper;
        $this->valueFactory = $valueFactory;
        $this->configGetter = $configGetter;
        $this->configurable = $configurable;
        $this->state = $state;
        $this->appEmulation = $appEmulation;
    }

    /**
     * @param Field $field
     * @param \Magento\Framework\GraphQl\Query\Resolver\ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return \Magento\Framework\GraphQl\Query\Resolver\Value
     */
    public function resolve(Field $field, $context, ResolveInfo $info, array $value = null, array $args = null)
    {
        if (!isset($value['model'])) {
            $result = function () {
                return null;
            };
            return $this->valueFactory->create($result);
        }
        $this->configurable->setNameInLayout('product.info.options.configurable');
        $this->configurable->setProduct($value['model']);
        try {
            $data = $this->state->emulateAreaCode(Area::AREA_FRONTEND, [$this, 'getContent'], [$value['store_id']]);
            $this->appEmulation->stopEnvironmentEmulation();
            $data = $this->prepareData($data);
        } catch (\Magento\Framework\Exception\NoSuchEntityException $e) {
            $data = [];
        }

        $result = function () use ($data) {
            return $data;
        };

        return $this->valueFactory->create($result);
    }

    /**
     * @param $storeId
     * @return array
     */
    public function getContent($storeId)
    {
        $this->appEmulation->startEnvironmentEmulation($storeId, Area::AREA_FRONTEND, true);

        return $this->configGetter->execute($this->configurable, true);
    }

    /**
     * @param array $data
     * @return array
     */
    private function prepareData($data)
    {
        $this->prepareTitles($data);
        $this->preparePreselect($data);
        $this->addMatrixData($data);

        return $data;
    }

    /**
     * @param array $data
     */
    private function prepareTitles(&$data)
    {
        if (isset($data[self::PRODUCT_PAGE_TYPE]['titles'])) {
            $titles = [];
            foreach ($data[self::PRODUCT_PAGE_TYPE]['titles'] as $title) {
                $titles[] = $title->render();
            }
            $data[self::PRODUCT_PAGE_TYPE]['titles'] = $titles;
        }
    }

    /**
     * @param array $data
     */
    private function preparePreselect(&$data)
    {
        foreach ([self::PRODUCT_PAGE_TYPE, self::PRODUCT_LISTING_TYPE] as $type) {
            if (isset($data[$type]['preselect']['attributes'])) {
                $attributes = [];
                foreach ($data[$type]['preselect']['attributes'] as $label => $id) {
                    $attributes[] = ['label' => $label, 'attribute_id' => $id];
                }
                $data[$type]['preselect']['attributes'] = $attributes;
            }
        }
    }

    /**
     * @param array $data
     */
    private function addMatrixData(&$data)
    {
        if (isset($data[self::PRODUCT_PAGE_TYPE])) {
            $data[self::PRODUCT_PAGE_TYPE]['matrix_data']['display_sku']
                = $this->helper->getModuleConfig('matrix/display_sku');
            $data[self::PRODUCT_PAGE_TYPE]['matrix_data']['available_qty']
                = $this->helper->getModuleConfig('matrix/available_qty');
            $data[self::PRODUCT_PAGE_TYPE]['matrix_data']['subtotal']
                = $this->helper->getModuleConfig('matrix/subtotal');
        }
    }
}
