<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package AJAX Shopping Cart for Magento 2
 */

namespace Amasty\Cart\Model;

use Amasty\Base\Model\ConfigProviderAbstract;
use Amasty\Cart\Model\Source\ConfirmPopup;
use Magento\Customer\Model\SessionFactory;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Store\Model\ScopeInterface;
use Magento\Tax\Model\Config;

class ConfigProvider extends ConfigProviderAbstract
{
    public const CONFIRM_POPUP_PATH = 'dialog_popup/confirm_popup';
    public const QUOTE_URL_KEY_PATH = 'amasty_request_quote/general/url_key';
    public const COMPARE_AJAX = 'general/compare';
    public const WISHLIST_AJAX = 'general/wishlist';

    /**
     * @var string
     */
    protected $pathPrefix = 'amasty_cart/';

    /**
     * @var SessionFactory
     */
    private $sessionFactory;

    public function __construct(
        ScopeConfigInterface $scopeConfig,
        SessionFactory $sessionFactory
    ) {
        parent::__construct($scopeConfig);
        $this->sessionFactory = $sessionFactory;
    }

    public function isMiniPage(): bool
    {
        return $this->getValue(self::CONFIRM_POPUP_PATH) == ConfirmPopup::MINI_PAGE;
    }

    public function getQuoteUrlKey(int $storeId = null): string
    {
        return (string)$this->scopeConfig->getValue(self::QUOTE_URL_KEY_PATH, ScopeInterface::SCOPE_STORE, $storeId);
    }

    public function isCompareAjax(int $storeId = null): bool
    {
        return $this->isSetFlag(self::COMPARE_AJAX, $storeId);
    }

    public function isWishlistAjax(int $storeId = null): bool
    {
        return $this->isSetFlag(self::WISHLIST_AJAX, $storeId)
            && !$this->scopeConfig->isSetFlag('wishlist/general/multiple_enabled')
            && $this->isCustomerLogged();
    }

    private function isCustomerLogged(): bool
    {
        return $this->sessionFactory->create()->isLoggedIn();
    }

    public function isDisplaySalesPricesInclTax(): bool
    {
        return $this->getScopeValueDisplaySubtotal() === Config::DISPLAY_TYPE_INCLUDING_TAX;
    }

    public function isDisplaySalesPricesExclTax(): bool
    {
        return $this->getScopeValueDisplaySubtotal() === Config::DISPLAY_TYPE_EXCLUDING_TAX;
    }

    public function isDisplaySalesPricesBoth(): bool
    {
        return $this->getScopeValueDisplaySubtotal() === Config::DISPLAY_TYPE_BOTH;
    }

    private function getScopeValueDisplaySubtotal(): int
    {
        return (int)$this->scopeConfig->getValue(Config::XML_PATH_DISPLAY_CART_SUBTOTAL);
    }
}
