<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package AJAX Shopping Cart for Magento 2
 */

namespace Amasty\Cart\Controller\Cart;

use Amasty\Cart\Helper\Data;
use Amasty\Cart\Model\ConfigProvider;
use Amasty\Cart\Model\Source\Section;
use Amasty\Cart\Model\SubtotalResolver;
use Magento\Checkout\Model\Sidebar;
use Magento\Framework\App\Action\Action;
use Magento\Framework\App\Action\Context;
use Magento\Framework\App\Response\Http;
use Magento\Framework\Data\Form\FormKey\Validator;
use Magento\Checkout\Model\Session;
use Magento\Framework\DataObjectFactory as ObjectFactory;
use Magento\Framework\Exception\LocalizedException;

class Update extends Action
{
    /**
     * @var string
     */
    protected $type = Section::CART;

    /**
     * @var Sidebar
     */
    private $sidebar;

    /**
     * @var Session
     */
    private $session;

    /**
     * @var ObjectFactory
     */
    private $objectFactory;

    /**
     * @var Validator
     */
    private $formKeyValidator;

    /**
     * @var Data
     */
    private $helper;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    /**
     * @var SubtotalResolver
     */
    private $subtotalResolver;

    public function __construct(
        Context $context,
        Sidebar $sidebar,
        Session $session,
        ObjectFactory $objectFactory,
        Data $helper,
        Validator $formKeyValidator,
        ConfigProvider $configProvider,
        SubtotalResolver $subtotalResolver
    ) {
        parent::__construct($context);
        $this->sidebar = $sidebar;
        $this->session = $session;
        $this->objectFactory = $objectFactory;
        $this->formKeyValidator = $formKeyValidator;
        $this->helper = $helper;
        $this->configProvider = $configProvider;
        $this->subtotalResolver = $subtotalResolver;
    }

    public function execute(): Http
    {
        try {
            $this->validateData();
            $itemId = (int)$this->getRequest()->getParam('item_id');
            $itemQty = (int)$this->getRequest()->getParam('item_qty');

            $this->getSidebar()->checkQuoteItem($itemId);
            $this->getSidebar()->updateQuoteItem($itemId, $itemQty);
            $quote = $this->getCheckoutSession()->getQuote();

            $resultObject = $this->objectFactory->create(
                [
                    'data' => [
                        'result' => [
                            'items'    => $quote->getItemsSummaryQty() . __(' items'),
                            'subtotalInclTax' => $this->subtotalResolver->getSubtotalHtml(
                                $this->getCheckoutSession(),
                                true
                            ),
                            'subtotalExclTax' => $this->subtotalResolver->getSubtotalHtml($this->getCheckoutSession())
                        ]
                    ]
                ]
            );
        } catch (LocalizedException $exception) {
            $resultObject = $this->objectFactory->create(
                [
                    'data' => [
                        'result' => [
                            'error' => $exception->getMessage()
                        ]
                    ]
                ]
            );
        } catch (\Exception $exception) {
            $resultObject = $this->objectFactory->create(
                [
                    'data' => [
                        'result' => [
                            'error' => __('We can\'t add this item to your shopping cart.')
                        ]
                    ]
                ]
            );
        }

        return $this->getResponse()->representJson(
            $this->helper->encode($resultObject->getResult())
        );
    }

    /**
     * @throws LocalizedException
     */
    protected function validateData(): void
    {
        if (!$this->formKeyValidator->validate($this->getRequest())) {
            throw new LocalizedException(
                __('We can\'t add this item to your shopping cart right now. Please reload the page.')
            );
        }

        $itemQty = (int)$this->getRequest()->getParam('item_qty');

        if ($itemQty <= 0) {
            throw new LocalizedException(
                __('We can\'t add this item to your shopping cart.')
            );
        }
    }

    public function getSidebar(): Sidebar
    {
        return $this->sidebar;
    }

    public function getCheckoutSession(): Session
    {
        return $this->session;
    }
}
