<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Amasty Blog GraphQl for Magento 2 (System)
 */

namespace Amasty\BlogGraphQl\Test\GraphQl;

use Amasty\Blog\Api\CategoryRepositoryInterface;
use Amasty\Blog\Api\PostRepositoryInterface;
use Amasty\Blog\Api\TagRepositoryInterface;
use Magento\Widget\Model\ResourceModel\Widget\Instance\CollectionFactory;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

class AmBlogRecentPostsWidgetTest extends GraphQlAbstract
{
    public const MAIN_QUERY_KEY = 'amBlogRecentPostsWidget';

    /**
     * @var PostRepositoryInterface
     */
    private $postRepository;

    /**
     * @var CollectionFactory
     */
    private $collectionFactory;

    /**
     * @var CategoryRepositoryInterface
     */
    private $categoryRepository;

    /**
     * @var TagRepositoryInterface
     */
    private $tagRepository;

    protected function setUp(): void
    {
        $objectManager = Bootstrap::getObjectManager();
        $this->postRepository = $objectManager->create(PostRepositoryInterface::class);
        $this->collectionFactory = $objectManager->get(CollectionFactory::class);
        $this->categoryRepository = $objectManager->get(CategoryRepositoryInterface::class);
        $this->tagRepository = $objectManager->get(TagRepositoryInterface::class);
    }

    /**
     * @group amasty_blog
     *
     * @magentoApiDataFixture Amasty_BlogGraphQl::Test/GraphQl/_files/widgets/recent_post_widget.php
     */
    public function testAmBlogRecentPostsWidgetBase(): void
    {
        $postOneUrlKey = 'post-test-one-graph-ql';
        $postThreeUrlKey = 'post-test-three-graph-ql';
        $categoryId = $this->categoryRepository->getByUrlKey('cat-graph-ql')->getCategoryId();
        $tagId = $this->tagRepository->getByUrlKey('tag-graph-one')->getTagId();
        $postOneId = $this->postRepository->getByUrlKey($postOneUrlKey)->getPostId();
        $postThreeId = $this->postRepository->getByUrlKey($postThreeUrlKey)->getPostId();
        $widgetId = (int)$this->collectionFactory->create()
            ->addFieldToFilter('title', 'Test Recent Post Widget Title')
            ->getFirstItem()->getId();

        $fields = [
            "amasty_widget_categories" => "$categoryId",
            "amasty_widget_tags" => "$tagId",
            "date_manner" => "passed",
            "display_date" => true,
            "display_short" => true,
            "header_text" => "Header Text Recent Post Test Graph",
            "items" => [
                [
                    "post_id" => $postOneId,
                    "url_key" => "$postOneUrlKey"
                ],
                [
                    "post_id" => $postThreeId,
                    "url_key" => "$postThreeUrlKey"
                ]
            ],
            "posts_limit" => 100001,
            "short_limit" => 256,
            "show_images" => true,
            "title" => "Test Recent Post Widget Title"
        ];

        $query = $this->getQuery($widgetId);
        $response = $this->graphQlQuery($query);

        $this->assertArrayHasKey(self::MAIN_QUERY_KEY, $response);
        $this->assertResponseFields($response[self::MAIN_QUERY_KEY], $fields);
    }

    /**
     * Returns GraphQl query string
     */
    private function getQuery(int $widgetId): string
    {
        return <<<QUERY
query {
    amBlogRecentPostsWidget(id: $widgetId) {
        amasty_widget_categories
        amasty_widget_tags
        date_manner
        display_date
        display_short
        header_text
        items {
            post_id
            url_key
        }
        posts_limit
        short_limit
        show_images
        title
    }
}
QUERY;
    }
}
