<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Amasty Blog GraphQl for Magento 2 (System)
 */

namespace Amasty\BlogGraphQl\Test\GraphQl;

use Amasty\Blog\Api\AuthorRepositoryInterface;
use Amasty\Blog\Api\CategoryRepositoryInterface;
use Amasty\Blog\Api\PostRepositoryInterface;
use Amasty\Blog\Api\TagRepositoryInterface;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

class AmBlogPostTest extends GraphQlAbstract
{
    public const MAIN_QUERY_KEY = 'amBlogPost';

    /**
     * @var PostRepositoryInterface
     */
    private $postRepository;

    /**
     * @var AuthorRepositoryInterface
     */
    private $authorRepository;

    /**
     * @var CategoryRepositoryInterface
     */
    private $categoryRepository;

    /**
     * @var TagRepositoryInterface
     */
    private $tagRepository;

    protected function setUp(): void
    {
        $objectManager = Bootstrap::getObjectManager();
        $this->postRepository = $objectManager->create(PostRepositoryInterface::class);
        $this->authorRepository = $objectManager->create(AuthorRepositoryInterface::class);
        $this->categoryRepository = $objectManager->create(CategoryRepositoryInterface::class);
        $this->tagRepository = $objectManager->create(TagRepositoryInterface::class);
    }

    /**
     * @group amasty_blog
     *
     * @magentoApiDataFixture Amasty_BlogGraphQl::Test/GraphQl/_files/posts.php
     */
    public function testAmBlogPostBase(): void
    {
        $postUrlKey = 'post-test-two-graph-ql';

        $authorId = (string)$this->authorRepository->getByUrlKey('author-graph-name-two')->getAuthorId();
        $categoryId = (string)$this->categoryRepository->getByUrlKey('cat-ch-t-graph-ql')->getCategoryId();
        $tagId = $this->tagRepository->getByUrlKey('tag-graph-two')->getTagId();
        $postId = $this->postRepository->getByUrlKey($postUrlKey)->getPostId();
        $postRelId = $this->postRepository->getByUrlKey('post-test-one-graph-ql')->getPostId();

        $fieldsResponse = [
            "author_id" => $authorId,
            "canonical_url" => "https://amasty.com/",
            "categories" => $categoryId,
            "comment_count" => 0,
            "comments_enabled" => "0",
            "display_short_content" => "1",
            "full_content" => "Full Test Two GraphQl Content",
            "grid_class" => "w1",
            "is_featured" => true,
            "list_thumbnail_alt" => "thumb_test_alt",
            "meta_description" => "Meta Graph Two Description",
            "meta_robots" => "index, follow",
            "meta_tags" => "Meta Graph Two Tag",
            "meta_title" => "Meta Graph Two Title",
            "notify_on_enable" => "0",
            "post_id" => $postId,
            "post_thumbnail_alt" => "thumb_post_test_alt",
            "related_post_ids" => $postRelId,
            "short_content" => "Short Test Two GraphQl Content",
            "status" => "2",
            "tag_ids" => $tagId,
            "title" => "Post Test Two GraphQl",
            "url_key" => "post-test-two-graph-ql",
            "user_define_publish" => "0",
            "views" => "1",
            "vote_count" => 0
        ];

        $query = $this->getQuery($postId, $postUrlKey);
        $response = $this->graphQlQuery($query);

        $this->assertArrayHasKey(self::MAIN_QUERY_KEY, $response);
        $this->assertResponseFields($response[self::MAIN_QUERY_KEY], $fieldsResponse);
        $this->assertArrayHasKey('list_thumbnail', $response[self::MAIN_QUERY_KEY]);
        $this->assertArrayHasKey('post_thumbnail', $response[self::MAIN_QUERY_KEY]);
        $this->assertArrayHasKey('relative_url', $response[self::MAIN_QUERY_KEY]);
        $this->assertArrayHasKey('published_at', $response[self::MAIN_QUERY_KEY]);
        $this->assertArrayHasKey('type', $response[self::MAIN_QUERY_KEY]);
        $this->assertStringContainsString(date("Y-m-d"), $response[self::MAIN_QUERY_KEY]['created_at']);
        $this->assertStringContainsString(date("Y-m-d"), $response[self::MAIN_QUERY_KEY]['updated_at']);
    }

    /**
     * Returns GraphQl query string
     */
    private function getQuery(int $id, string $urlKey): string
    {
        return <<<QUERY
query {
    amBlogPost(id: $id, urlKey: "$urlKey") {
        author_id
        canonical_url
        categories
        comment_count
        comments_enabled
        created_at
        display_short_content
        full_content
        grid_class
        is_featured
        list_thumbnail
        list_thumbnail_alt
        meta_description
        meta_robots
        meta_tags
        meta_title
        notify_on_enable
        post_id
        post_thumbnail
        post_thumbnail_alt
        published_at
        related_post_ids
        relative_url
        short_content
        status
        tag_ids
        title
        type
        updated_at
        url_key
        user_define_publish
        views
        vote_count
    }
}
QUERY;
    }
}
