<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Amasty Blog GraphQl for Magento 2 (System)
 */

namespace Amasty\BlogGraphQl\Test\GraphQl;

use Amasty\Blog\Api\PostRepositoryInterface;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\GraphQlAbstract;

class AmBlogCommentsTest extends GraphQlAbstract
{
    public const MAIN_QUERY_KEY = 'amBlogComments';

    /**
     * @var PostRepositoryInterface
     */
    private $postRepository;

    /**
     * @var CustomerRepositoryInterface
     */
    private $customerRepository;

    protected function setUp(): void
    {
        $objectManager = Bootstrap::getObjectManager();
        $this->postRepository = $objectManager->create(PostRepositoryInterface::class);
        $this->customerRepository = $objectManager->create(CustomerRepositoryInterface::class);
    }

    /**
     * @group amasty_blog
     *
     * @magentoApiDataFixture Amasty_BlogGraphQl::Test/GraphQl/_files/posts_comments.php
     */
    public function testAmBlogCommentsList(): void
    {
        $result = false;
        $customerId = $this->customerRepository->get('new_customer@example.com')->getId();
        $postTwoId = $this->postRepository->getByUrlKey('post-test-one-graph-ql')->getPostId();
        $postThreeId = $this->postRepository->getByUrlKey('post-test-three-graph-ql')->getPostId();

        $fieldsComment = [
            "customer_id" => "$customerId",
            "message" => "First Test Graph Message Comment Post One",
            "name" => "John",
            "post_id" => "$postTwoId",
            "status" => "2",
            "store_id" => "1"
        ];

        $query = $this->getQuery($postTwoId, 'list');
        $response = $this->graphQlQuery($query);

        $this->assertArrayHasKey(self::MAIN_QUERY_KEY, $response);
        $this->assertEquals(3, count($response[self::MAIN_QUERY_KEY]['items']));

        foreach ($response[self::MAIN_QUERY_KEY]['items'] as $item) {
            if ($item['email'] == 'new_customer@example.com') {
                $this->assertArrayHasKey('post', $item);
                $this->assertArrayHasKey('reply_to', $item);
                $this->assertResponseFields($item, $fieldsComment);
                $this->assertStringContainsString(date("Y-m-d"), $item['created_at']);
                $this->assertStringContainsString(date("Y-m-d"), $item['updated_at']);

                $result = true;
            }
        }

        $this->assertTrue($result);

        $querySecondPost = $this->getQuery($postThreeId, 'list');
        $responseSecondPost = $this->graphQlQuery($querySecondPost);

        $this->assertArrayHasKey(self::MAIN_QUERY_KEY, $responseSecondPost);
        $this->assertEquals(1, count($responseSecondPost[self::MAIN_QUERY_KEY]['items']));
    }

    /**
     * @group amasty_blog
     *
     * @magentoApiDataFixture Amasty_BlogGraphQl::Test/GraphQl/_files/posts_comments.php
     */
    public function testAmBlogCommentsRecent(): void
    {
        $result = false;
        $postId = $this->postRepository->getByUrlKey('post-test-three-graph-ql')->getPostId();

        $fieldsComment = [
            "message" => "Reply Test Graph Message Comment Post Three",
            "name" => "Graph Two",
            "post_id" => "$postId",
            "status" => "2",
            "store_id" => "1"
        ];

        $query = $this->getQuery(0, 'recent');
        $response = $this->graphQlQuery($query);

        $this->assertArrayHasKey(self::MAIN_QUERY_KEY, $response);
        $this->assertGreaterThanOrEqual(2, count($response[self::MAIN_QUERY_KEY]['items']));

        foreach ($response[self::MAIN_QUERY_KEY]['items'] as $item) {
            if ($item['email'] == 'graph_two@example.com') {
                $this->assertArrayHasKey('post', $item);
                $this->assertArrayHasKey('reply_to', $item);
                $this->assertArrayHasKey('customer_id', $item);
                $this->assertResponseFields($item, $fieldsComment);
                $this->assertStringContainsString(date("Y-m-d"), $item['created_at']);
                $this->assertStringContainsString(date("Y-m-d"), $item['updated_at']);

                $result = true;
            }
        }

        $this->assertTrue($result, 'Not returns all items.');
    }

    /**
     * Returns GraphQl query string
     */
    private function getQuery(int $postId, string $type): string
    {
        return <<<QUERY
query {
    amBlogComments(postId: $postId, type: "$type") {
        items {
            comment_id
            created_at
            customer_id
            email
            message
            name
            post {
                url_key
            }
            post_id
            reply_to
            status
            store_id
            updated_at
        }
    }
}
QUERY;
    }
}
