<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Amasty Blog GraphQl for Magento 2 (System)
 */

namespace Amasty\BlogGraphQl\Model\UrlRewrite;

use Amasty\Blog\Model\ConfigProvider;

class UrlKeyMatcher
{
    /**
     * Regex pattern to find required url key
     *
     * @var string
     */
    private const REGEX_URL_KEY_PART = "(?'url_key'.*)$";

    /**
     * @var string|null
     */
    private $entityIdentifier = null;

    /**
     * @var ConfigProvider
     */
    private $configProvider;

    public function __construct(
        ConfigProvider $configProvider,
        ?string $entityIdentifier = null
    ) {
        $this->configProvider = $configProvider;
        $this->entityIdentifier = $entityIdentifier;
    }

    public function match(string $url) : ?string
    {
        // phpcs:disable Magento2.Functions.DiscouragedFunction.Discouraged
        $parsedUrl = parse_url($url)['path'] ?? $url;
        $postfix = preg_quote($this->configProvider->getBlogPostfix(), '/');
        $parsedUrl = preg_replace("/$postfix$/i", '', $parsedUrl);
        preg_match($this->buildRegexPattern(), $parsedUrl, $matches);

        return $matches['url_key'] ?? null;
    }

    /**
     * Seo route + entity identifier (if necessary) + url_key (till string ends)
     * Example:
     * - /blog\/(?'url_key'.*)$/
     * - /blog\/category\/(?'url_key'.*)$/
     *
     * @return string
     */
    private function buildRegexPattern(): string
    {
        $pattern = '/' . $this->configProvider->getSeoRoute() . '\/';
        $pattern .= $this->entityIdentifier ? $this->entityIdentifier . '\/' : '';
        $pattern .= self::REGEX_URL_KEY_PART . '/i';

        return $pattern;
    }
}
