<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Amasty Blog GraphQl for Magento 2 (System)
 */

namespace Amasty\BlogGraphQl\Model\Resolver\Widget;

use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;

abstract class AbstractWidget implements ResolverInterface
{
    /**
     * @var int
     */
    protected $storeId;

    /**
     * @var \Magento\Widget\Model\ResourceModel\Widget\Instance\Collection
     */
    protected $widgetCollection;

    public function __construct(
        \Magento\Widget\Model\ResourceModel\Widget\Instance\Collection $widgetCollection
    ) {
        $this->widgetCollection = $widgetCollection;
    }

    /**
     * @param Field $field
     * @param \Magento\Framework\GraphQl\Query\Resolver\ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return array|\Magento\Framework\GraphQl\Query\Resolver\Value|mixed
     * @throws \Exception
     */
    public function resolve(
        Field $field,
        $context,
        ResolveInfo $info,
        array $value = null,
        array $args = null
    ) {
        $widget = isset($args['id']) ? $this->widgetCollection->getItemById($args['id']) : null;
        $storeId = (int)$context->getExtensionAttributes()->getStore()->getId();
        $this->setStoreId($storeId);
        if ($widget) {
            if (!$this->validateStore($widget, $storeId)) {
                throw new GraphQlNoSuchEntityException(__('Wrong parameter storeId.'));
            }

            $data = $widget->getWidgetParameters();
            $data['title'] = $widget->getData('title');
        } else {
            $data = $this->generateDataFromBlock();
        }

        $data['items'] = [];
        foreach ($this->getItemCollection($data, $storeId) as $key => $item) {
            $data['items'][$key] = $item->getData();
            $data['items'][$key]['model'] = $item;
        }

        return $this->convertData($data, $context);
    }

    /**
     * @param array $data
     * @param int $storeId
     * @return mixed
     */
    abstract protected function getItemCollection(array $data, $storeId = 0);

    /**
     * @return array
     */
    abstract protected function generateDataFromBlock();

    /**
     * @param array $data
     * @param $data
     * @return array
     */
    protected function convertData(array $data, $context)
    {
        return $data;
    }

    /**
     * @param $widget
     * @param int $storeId
     * @return bool
     */
    private function validateStore($widget, int $storeId)
    {
        return in_array(\Magento\Store\Model\Store::DEFAULT_STORE_ID, $widget->getStoreIds())
            || in_array($storeId, $widget->getStoreIds());
    }

    /**
     * @return int
     */
    public function getStoreId(): int
    {
        return $this->storeId;
    }

    /**
     * @param int $storeId
     */
    public function setStoreId(int $storeId): void
    {
        $this->storeId = $storeId;
    }

    /**
     * @param string $src
     * @param $context
     *
     * @return string|string[]
     */
    protected function getRelativePath(string $src, $context)
    {
        $baseUrl = trim($context->getExtensionAttributes()->getStore()->getBaseUrl(), '/');

        return str_replace($baseUrl, '', $src);
    }

    /**
     * @param array $item
     * @param $context
     *
     * @return array
     */
    protected function convertPostItem(array $item, $context)
    {
        $item['categories'] = implode(', ', $item['categories']);
        $item['tags'] = implode(', ', $item['tag_ids']);
        $item['post_thumbnail'] = $this->getRelativePath((string)$item['model']->getPostThumbnailSrc(), $context);
        $item['list_thumbnail'] = $this->getRelativePath((string)$item['model']->getListThumbnailSrc(), $context);

        return $item;
    }
}
